// AI Parlay Builder WordPress Plugin JavaScript

(function($) {
    'use strict';
    
    let selectedSports = ['basketball_nba'];
    let selectedRisk = 'balanced';
    let selectedLegs = 3;
    let selectedBetTypes = ['moneyline', 'spread'];
    let dateRange = 'today';
    let availableSports = [];
    
    $(document).ready(function() {
        initializeParlayBuilder();
    });
    
    function initializeParlayBuilder() {
        loadSports();
        bindEvents();
        updateLegsDisplay();
    }
    
    function bindEvents() {
        // Risk level selection
        $(document).on('click', '.risk-btn', function() {
            $('.risk-btn').removeClass('active');
            $(this).addClass('active');
            selectedRisk = $(this).data('risk');
        });
        
        // Number of legs slider
        $(document).on('input', '#num-legs', function() {
            selectedLegs = parseInt($(this).val());
            updateLegsDisplay();
        });
        
        // Bet types checkboxes
        $(document).on('change', '.bet-types input[type="checkbox"]', function() {
            updateBetTypes();
        });
        
        // Date range selection
        $(document).on('change', '#date-range', function() {
            dateRange = $(this).val();
        });
        
        // Generate parlay button
        $(document).on('click', '#generate-parlay', function() {
            generateParlay();
        });
        
        // Sports selection (will be bound after sports are loaded)
        $(document).on('change', '.sport-checkbox', function() {
            updateSelectedSports();
        });
    }
    
    function updateLegsDisplay() {
        $('#legs-display').text(selectedLegs);
    }
    
    function updateBetTypes() {
        selectedBetTypes = [];
        $('.bet-types input[type="checkbox"]:checked').each(function() {
            selectedBetTypes.push($(this).val());
        });
    }
    
    function updateSelectedSports() {
        selectedSports = [];
        $('.sport-checkbox:checked').each(function() {
            selectedSports.push($(this).val());
        });
    }
    
    function loadSports() {
        $.ajax({
            url: aiParlay.ajax_url,
            type: 'POST',
            data: {
                action: 'get_sports',
                nonce: aiParlay.nonce
            },
            success: function(response) {
                if (response.success) {
                    availableSports = response.data;
                    renderSports(response.data);
                } else {
                    showError('Failed to load sports: ' + response.data);
                }
            },
            error: function() {
                showError('Failed to load sports data');
            }
        });
    }
    
    function renderSports(sports) {
        let html = '';
        sports.forEach(function(sport) {
            const checked = selectedSports.includes(sport.key) ? 'checked' : '';
            const activeClass = checked ? 'active' : '';
            
            html += `
                <div class="sport-option ${activeClass}">
                    <input type="checkbox" class="sport-checkbox" value="${sport.key}" ${checked} id="sport-${sport.key}">
                    <label for="sport-${sport.key}">${sport.title}</label>
                </div>
            `;
        });
        
        $('#sports-selection').html(html);
    }
    
    function generateParlay() {
        if (selectedSports.length === 0) {
            showError('Please select at least one sport');
            return;
        }
        
        if (selectedBetTypes.length === 0) {
            showError('Please select at least one bet type');
            return;
        }
        
        showLoading();
        
        $.ajax({
            url: aiParlay.ajax_url,
            type: 'POST',
            data: {
                action: 'generate_parlay',
                nonce: aiParlay.nonce,
                sports: selectedSports.join(','),
                risk_level: selectedRisk,
                num_legs: selectedLegs,
                bet_types: selectedBetTypes.join(','),
                date_range: dateRange
            },
            success: function(response) {
                hideLoading();
                if (response.success) {
                    renderParlay(response.data);
                } else {
                    showError('Failed to generate parlay: ' + response.data);
                }
            },
            error: function() {
                hideLoading();
                showError('Failed to generate parlay');
            }
        });
    }
    
    function showLoading() {
        $('#loading-state').show();
        $('#parlay-suggestions').removeClass('show');
        $('#empty-state').hide();
        $('#generate-parlay').prop('disabled', true).text('Generating...');
    }
    
    function hideLoading() {
        $('#loading-state').hide();
        $('#generate-parlay').prop('disabled', false).text('🚀 Generate AI Parlays');
    }
    
    function showError(message) {
        hideLoading();
        alert(message); // In production, you might want a nicer error display
    }
    
    function renderParlay(parlayData) {
        $('#empty-state').hide();
        
        const html = `
            <div class="parlay-card">
                <div class="parlay-header-card">
                    <div class="parlay-header-top">
                        <div class="parlay-info">
                            <h3>AI Parlay #1</h3>
                            <span class="badge badge-risk">${selectedRisk.toUpperCase()}</span>
                            <span class="badge badge-confidence">Confidence: ${parlayData.confidence_score}/10</span>
                        </div>
                        <div class="parlay-odds">
                            <div class="odds-display">${formatOdds(parlayData.total_odds)}</div>
                            <div class="probability-display">${parlayData.implied_probability}% probability</div>
                        </div>
                    </div>
                    
                    <div class="payout-grid">
                        <div class="payout-item">
                            <div class="payout-amount">$${calculatePayout(parlayData.total_odds, 100).toFixed(2)}</div>
                            <div class="payout-stake">$100 Bet Payout</div>
                        </div>
                        <div class="payout-item">
                            <div class="payout-amount">$${calculatePayout(parlayData.total_odds, 50).toFixed(2)}</div>
                            <div class="payout-stake">$50 Bet Payout</div>
                        </div>
                        <div class="payout-item">
                            <div class="payout-amount">$${calculatePayout(parlayData.total_odds, 25).toFixed(2)}</div>
                            <div class="payout-stake">$25 Bet Payout</div>
                        </div>
                    </div>
                </div>
                
                <div class="parlay-body">
                    <div class="selections-list">
                        ${renderSelections(parlayData.selections)}
                    </div>
                    
                    <div class="analysis-section">
                        <h4 class="analysis-title">🧠 AI Analysis</h4>
                        <p class="analysis-content">${parlayData.ai_rationale}</p>
                    </div>
                    
                    <div class="risk-section">
                        <h4 class="analysis-title">⚠️ Risk Assessment</h4>
                        <p class="analysis-content">${parlayData.risk_assessment}</p>
                    </div>
                    
                    <div class="parlay-actions">
                        <button class="btn btn-primary">🎯 Bet This Parlay</button>
                    </div>
                </div>
            </div>
        `;
        
        $('#parlay-suggestions').html(html).addClass('show');
    }
    
    function renderSelections(selections) {
        let html = '';
        
        selections.forEach(function(selection, index) {
            const gameDate = new Date(selection.commence_time).toLocaleDateString();
            const sportTitle = getSportTitle(selection.sport);
            
            html += `
                <div class="selection-item">
                    <div class="selection-header">
                        <div class="selection-teams">
                            <div class="selection-number">${index + 1}</div>
                            <div class="teams-info">
                                <h4>${selection.away_team} @ ${selection.home_team}</h4>
                                <p>${gameDate} • ${sportTitle}</p>
                            </div>
                        </div>
                        <div class="selection-odds">
                            <div class="odds-value">${formatOdds(selection.odds)}</div>
                            <div class="bet-type">${selection.bet_type}</div>
                        </div>
                    </div>
                    
                    <div class="ai-pick">
                        <div class="pick-label">AI Pick:</div>
                        <div class="pick-details">
                            <span class="pick-team">${selection.selected_team}</span>
                            ${selection.line ? ` ${selection.line > 0 ? '+' : ''}${selection.line}` : ''}
                            <span style="color: #d1d5db;"> (${selection.bet_type})</span>
                        </div>
                    </div>
                </div>
            `;
        });
        
        return html;
    }
    
    function getSportTitle(sportKey) {
        const sport = availableSports.find(s => s.key === sportKey);
        return sport ? sport.title : sportKey.toUpperCase();
    }
    
    function formatOdds(odds) {
        return odds > 0 ? `+${odds}` : `${odds}`;
    }
    
    function calculatePayout(odds, stake) {
        if (odds > 0) {
            return stake + (stake * odds / 100);
        } else {
            return stake + (stake * 100 / Math.abs(odds));
        }
    }
    
    // Update sports selection display when checkboxes change
    $(document).on('change', '.sport-checkbox', function() {
        const $option = $(this).closest('.sport-option');
        if ($(this).is(':checked')) {
            $option.addClass('active');
        } else {
            $option.removeClass('active');
        }
        updateSelectedSports();
    });
    
})(jQuery);