<?php
/**
 * Campaign HTML
 *
 * @package Popup Anything on Click Pro
 * @since 2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if( ! class_exists( 'WP_List_Table' ) ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

class Paoc_Pro_Campaign extends WP_List_Table {

	var $prefix, $per_page, $campaign_count, $redirect_url;

	// Construct
	function __construct() {

		$this->prefix		= PAOC_PRO_META_PREFIX;
		$this->per_page		= apply_filters( 'paoc_pro_campaign_per_page', 15 ); // Per page
		$this->redirect_url	= add_query_arg( array('post_type' => PAOC_PRO_POST_TYPE, 'page' => 'paoc-campaign'), admin_url('edit.php') );

		// Set parent defaults
		parent::__construct( array(
								'singular'  => 'paoc_pro_campaign',	// singular name of the listed records
								'plural'    => 'paoc_pro_campaign',	// plural name of the listed records
								'ajax'      => true					// does this table support ajax?
							));
	}

	/**
	 * Display Columns
	 * 
	 * Handles which columns to show in table
	 * 
	 * @since 2.0
	 */
	function get_columns() {
		$columns = array(
			'cb'			=> '<input type="checkbox" />', // Render a checkbox instead of text
			'title'			=> esc_html__('Title', 'popup-anything-on-click'),
			'enable'		=> esc_html__('Enable', 'popup-anything-on-click'),
			'popup_ids'		=> esc_html__('Popup IDs', 'popup-anything-on-click'),
			'created_date'	=> esc_html__('Created Date', 'popup-anything-on-click'),
		);
		return $columns;
	}

	/**
	 * Displaying campaign data
	 * 
	 * Does prepare the data for displaying the campaign list in the table.
	 * 
	 * @since 2.0
	 */
	function paoc_pro_campaign_lists() {

		// Taking some variable
		$campaigns	= array();
		$orderby    = isset( $_GET['orderby'] ) ? urldecode( $_GET['orderby'] ) : 'created_date';
		$order      = isset( $_GET['order'] )   ? $_GET['order']                : 'DESC';
		$args       = array(
						'limit'		=> $this->per_page,
						'orderby'	=> $orderby,
						'order'		=> $order,
						'search'	=> isset( $_GET['s'] ) ? $_GET['s'] : '',
					);

		// Get Campaign Data
		$campaign_data_arr		= paoc_pro_get_campaigns( $args );
		$campaign_data			= $campaign_data_arr['data'];
		$this->campaign_count	= $campaign_data_arr['count'];

		if( ! empty( $campaign_data ) ) {
			foreach ( $campaign_data as $campaign_key => $campaign_val ) {

				// Taking some variable
				$popup_data			= array();
				$campaign_enable	= ! empty( $campaign_val->enable )		? '<i class="paoc-dashicons-icon dashicons dashicons-yes paoc-dashicons-icon-success" title="'.esc_attr__('Campaign is active.', 'popup-anything-on-click').'"></i>' : '<i class="paoc-dashicons-icon dashicons dashicons-no-alt paoc-dashicons-icon-error" title="'.esc_attr__('Campaign is inactive.', 'popup-anything-on-click').'"></i>';
				$get_popup_ids		= ! empty( $campaign_val->popup_ids )	? explode( ',', $campaign_val->popup_ids ) : '';

				// Add link for particular popup id
				if( ! empty( $get_popup_ids ) ) {
					foreach ( $get_popup_ids as $popup_key => $popup_id ) {
						$popup_data[] = '<a href="'.esc_url( get_edit_post_link( $popup_id ) ).'" target="_blank">'.esc_html( $popup_id ).'</a>';
					}
				}

				$popup_id = implode( ', ', $popup_data );

				$campaigns[$campaign_key]['id'] 			= $campaign_val->id;
				$campaigns[$campaign_key]['title'] 			= $campaign_val->title;
				$campaigns[$campaign_key]['enable'] 		= $campaign_enable;
				$campaigns[$campaign_key]['popup_ids'] 		= $popup_id;
				$campaigns[$campaign_key]['popup_id'] 		= $campaign_val->popup_ids;
				$campaigns[$campaign_key]['created_date']	= $campaign_val->created_date;
			}
		}

		return $campaigns;
	}

	function column_default( $item, $column_name ) {
		return wp_kses_post( $item[$column_name] );
	}

	/**
	 * Handles checkbox HTML
	 * 
	 * @since 2.0
	 **/
	function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			$this->_args['singular'],  // Let's simply repurpose the table's singular label ("paoc-emails")
			$item['id']                // The value of the checkbox should be the record's id
		);
	}

	/**
	 * Manage Campaign Column
	 *
	 * @since 2.0
	 */
	function column_title( $item ) {

		$paged 				= isset( $_GET['paged'] ) ? $_GET['paged'] : false;
		$page_url			= add_query_arg( array( 'paged' => $paged ), $this->redirect_url );
		$edit_campaign 		= add_query_arg( array( 'action' => 'edit', 'campaign_id' => $item['id'] ), $page_url );
		$delete_campaign 	= add_query_arg( array( 'action' => 'delete', 'paoc_pro_campaign[]' => $item['id'], '_wpnonce' => wp_create_nonce('bulk-paoc_pro_campaign') ), $page_url );

		$actions['edit']	= sprintf('<a class="paoc-action-link" href="%s">'.esc_html__('Edit', 'popup-anything-on-click').'</a>', $edit_campaign );
		$actions['delete']	= sprintf('<a class="paoc-confirm paoc-action-link" href="%s">'.esc_html__('Delete', 'popup-anything-on-click').'</a>', $delete_campaign );

		// Return contents
		return sprintf('%1$s %2$s',
		   /*%1$s*/ $item['title'],
		   /*%2$s*/ $this->row_actions( $actions )
		);
	}

	/**
	 * Sortable Columns
	 *
	 * Handles soratable columns of the table
	 * 
	 * @since 2.0
	 */
	function get_sortable_columns() {

		$sortable_columns = array(
			'title'    		=> array('title', false),
			'created_date'  => array('created_date', false)
		);
		return $sortable_columns;
	}

	/**
	 * Bulk actions field
	 *
	 * Handles Bulk Action combo box values
	 * 
	 * @since 2.0
	 */
	function get_bulk_actions() {
		$actions = array(
						'delete' => esc_html__('Delete', 'popup-anything-on-click'),
					);
		return $actions;
	}

	/**
	 * Message to show when no records in database table
	 *
	 * @since 2.0
	 */
	function no_items() {
		echo esc_html__('No campaign found.', 'popup-anything-on-click');
	}

	/**
	 * Prepare Items for Campaign listing
	 * 
	 * @since 2.0
	 **/
	function prepare_items() {
		
		// Get how many records per page to show
		$per_page	= $this->per_page;
		
		// Get All, Hidden, Sortable columns
		$columns	= $this->get_columns();
		$hidden		= array();
		$sortable	= $this->get_sortable_columns();

		// Get final column header
		$this->_column_headers = array( $columns, $hidden, $sortable );

		// Get current page number
		$current_page = $this->get_pagenum();

		// Get total count
		$total_items = $this->campaign_count;

		// Get page items
		$this->items = $this->paoc_pro_campaign_lists();

		// Register pagination options and calculations.
		$this->set_pagination_args( array(
										'total_items' => $total_items,                  // Calculate the total number of items
										'per_page'    => $per_page,                     // Determine how many items to show on a page
										'total_pages' => ceil($total_items / $per_page)	// Calculate the total number of pages
									));
	}
}

$campaign_obj = new Paoc_Pro_Campaign();
$campaign_obj->prepare_items();
?>

<div class="wrap paoc-campaign-wrap">

	<h2>
		<?php esc_html_e( 'Popup Anything Pro - A/B Testing Campaign', 'popup-anything-on-click' ); ?>
		<a href="<?php echo esc_url( PAOC_PRO_ADD_CAMPAIGN_LINK ); ?>" class="page-title-action"><?php esc_html_e('Add New', 'popup-anything-on-click')?></a>
	</h2>

	<?php if( ! empty($_GET['message']) && $_GET['message'] == 'delete' ) {
		paoc_pro_display_message( 'update', __('Campaign deleted successfully.', 'popup-anything-on-click') );
	} ?>

	<form id="paoc-campaign" class="paoc-campaign" method="get" action="">

		<input type="hidden" name="post_type" value="<?php echo esc_attr( $_REQUEST['post_type'] ); ?>" />
		<input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>" />

		<?php

			$campaign_obj->search_box( esc_html__( 'Search', 'popup-anything-on-click' ), 'popup-anything-on-click' );

			$campaign_obj->views();	// Showing sorting links on the top of the list

			$campaign_obj->display();	// Display all the campaigns
		?>
	</form><!-- end .paoc-campaign -->
</div>