<?php
/**
 * Admin Class
 *
 * Handles the Admin side functionality of plugin
 *
 * @package Popup Anything on Click Pro
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Paoc_Pro_Admin {

	function __construct() {

		// Action to add admin menu
		add_action( 'admin_menu', array($this, 'paoc_pro_register_menu') );

		// Action to add metabox
		add_action( 'add_meta_boxes', array($this, 'paoc_pro_post_sett_metabox') );

		// Action to save metabox
		add_action( 'save_post', array($this, 'paoc_pro_save_metabox_value') );

		// Action to display admin notice
		add_action( 'admin_notices', array($this, 'paoc_pro_admin_notices') );

		// Action to delete popup post data
		add_action( 'before_delete_post', array($this, 'paoc_pro_delete_popup_data'), 10, 2 );

		// Action to register plugin settings
		add_action ( 'admin_init', array($this, 'paoc_pro_admin_processes') );

		// Action to add custom column to Slider listing
		add_filter( 'manage_'.PAOC_PRO_POST_TYPE.'_posts_columns', array($this, 'paoc_pro_manage_posts_columns') );

		// Action to add custom column data to Slider listing
		add_action('manage_'.PAOC_PRO_POST_TYPE.'_posts_custom_column', array($this, 'paoc_pro_post_columns_data'), 10, 2);

		// Filter to add row data
		add_filter( 'post_row_actions', array($this, 'paoc_pro_post_row_action'), 10, 2 );

		// Filter to add plugin links
		add_filter( 'plugin_row_meta', array( $this, 'paoc_pro_plugin_row_meta' ), 10, 2 );

		// Action to Add Duplicate Post Link in post edit sidebar
		add_action( 'post_submitbox_misc_actions', array($this, 'paoc_pro_duplicate_post_link') );

		// Action to Duplicate the post
		add_action( 'admin_action_paoc_pro_duplicate_post', array($this, 'paoc_pro_duplicate_post') );

		// Action to get post suggestion
		add_action( 'wp_ajax_paoc_pro_post_title_sugg', array($this, 'paoc_pro_post_title_sugg') );

		// Action to flush the popup stats
		add_action( 'wp_ajax_paoc_pro_flush_popup_stats', array($this, 'paoc_pro_flush_popup_stats') );
	}

	/**
	 * Function to add menu
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.2.4
	 */
	function paoc_pro_register_menu() {

		// Register Form Entries page
		add_submenu_page( 'edit.php?post_type='.PAOC_PRO_POST_TYPE, __('Form Entries - Popup Anything Pro', 'popup-anything-on-click'), __('Form Entries', 'popup-anything-on-click'), 'manage_options', 'paoc-form-entries', array($this, 'paoc_pro_form_entries_page') );

		// Register Report page
		add_submenu_page( 'edit.php?post_type='.PAOC_PRO_POST_TYPE, __('Reports - Popup Anything Pro', 'popup-anything-on-click'), __('Reports', 'popup-anything-on-click'), 'manage_options', 'paoc-reports', array($this, 'paoc_pro_report_page') );

		// Register Export page
		add_submenu_page( 'edit.php?post_type='.PAOC_PRO_POST_TYPE, __('Tools - Popup Anything Pro', 'popup-anything-on-click'), __('Tools', 'popup-anything-on-click'), 'manage_options', 'paoc-tools', array($this, 'paoc_pro_tools_page') );

		// Register Setting page
		add_submenu_page( 'edit.php?post_type='.PAOC_PRO_POST_TYPE, __('Settings - Popup Anything On Click Pro ', 'popup-anything-on-click'), __('Settings', 'popup-anything-on-click'), 'manage_options', 'paoc-pro-settings', array($this, 'paoc_pro_settings_page') );
	}

	/**
	 * Function to handle the form entries page html
	 *
	 * @package Popup Anything on Click Pro
	 * @since 2.0
	 */
	function paoc_pro_form_entries_page() {

		if( isset( $_GET['action'] ) && $_GET['action'] == 'view' && ! empty( $_GET['entry_id'] ) ) {
			include_once( PAOC_PRO_DIR . '/includes/admin/form/form-entries-view.php' );
		} else {
			include_once( PAOC_PRO_DIR . '/includes/admin/form/form-entries.php' );
		}
	}

	/**
	 * Function to handle the report page html
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 2.0
	 */
	function paoc_pro_report_page() {
		include_once( PAOC_PRO_DIR . '/includes/admin/report/reports.php' );
	}

	/**
	 * Function to handle the tools page html
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 2.0
	 */
	function paoc_pro_tools_page() {
		include_once( PAOC_PRO_DIR . '/includes/admin/tools/export.php' );
	}

	/**
	 * Getting Started Page Html
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.3
	 */
	function paoc_pro_settings_page() {
		include_once( PAOC_PRO_DIR . '/includes/admin/settings/settings.php' );
	}

	/**
	 * Post Settings Metabox
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.0
	 */
	function paoc_pro_post_sett_metabox() {

		$paoc_post_types = paoc_pro_get_option( 'post_types', array() );

		// Add metabox in popup posts
		add_meta_box( 'paoc-popup-sett', __( 'Popup Anything Pro - Settings', 'popup-anything-on-click' ), array($this, 'paoc_pro_meta_box_content'), PAOC_PRO_POST_TYPE, 'normal', 'high' );

		// Add metabox in popup Report
		add_meta_box( 'paoc-popup-report', __( 'Popup Report', 'popup-anything-on-click' ), array($this, 'paoc_pro_report_meta_box_content'), PAOC_PRO_POST_TYPE, 'side', 'default' );

		// Add metabox for selected post type
		if( $paoc_post_types ) {
			add_meta_box( 'paoc-post-sett', __( 'Popup Anything On Click Pro', 'popup-anything-on-click' ), array($this, 'paoc_pro_sett_mb_content'), $paoc_post_types, 'normal', 'default' );
		}
	}

	/**
	 * Function to handle metabox content
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 2.0
	 */
	function paoc_pro_meta_box_content() {
		include_once( PAOC_PRO_DIR .'/includes/admin/metabox/post-sett-metabox.php');
	}

	/**
	 * Function to handle metabox content
	 * 
	 * @since 2.0
	 */
	function paoc_pro_report_meta_box_content() {
		include_once( PAOC_PRO_DIR .'/includes/admin/metabox/report-metabox.php');
	}

	/**
	 * Post type metabox HTML
	 * 
	 * @since 2.0
	 */
	function paoc_pro_sett_mb_content() {
		include_once( PAOC_PRO_DIR .'/includes/admin/metabox/paoc-sett-metabox.php');
	}

	/**
	 * Function to save metabox values
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.0
	 */
	function paoc_pro_save_metabox_value( $post_id ) {

		global $post_type;

		$prefix		= PAOC_PRO_META_PREFIX; // Taking metabox prefix
		$post_types	= paoc_pro_get_option( 'post_types', array() );

		// Individual Post Meta
		if ( ! ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )						// Check Autosave
		&& ( isset( $_POST['post_ID'] ) && absint( $_POST['post_ID'] ) == $post_id )	// Check Revision
		&& ( in_array( $post_type, $post_types ) )										// Check if correct post type
		&& ( current_user_can( 'edit_post', $post_id ) ) )
		{

			$welcome_popup	= isset( $_POST[$prefix.'welcome_popup'] )	? paoc_pro_clean_number( $_POST[$prefix.'welcome_popup'], '', 'number' )	: '';
			$exit_popup		= isset( $_POST[$prefix.'exit_popup'] )		? paoc_pro_clean_number( $_POST[$prefix.'exit_popup'], '', 'number' )		: '';
			$general_popup	= isset( $_POST[$prefix.'general_popup'] )	? paoc_pro_clean( $_POST[$prefix.'general_popup'] )							: array();

			update_post_meta( $post_id, $prefix.'welcome_popup', $welcome_popup );
			update_post_meta( $post_id, $prefix.'exit_popup', $exit_popup );
			update_post_meta( $post_id, $prefix.'general_popup', $general_popup );
		}

		// Popup Meta
		if ( ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE )							// Check Autosave
		|| ( ! isset( $_POST['post_ID'] ) || absint( $_POST['post_ID'] ) != $post_id )	// Check Revision
		|| ( $post_type !=  PAOC_PRO_POST_TYPE )										// Check if current post type is supported.
		|| ( ! current_user_can( 'edit_post', $post_id ) ) )
		{
			return $post_id;
		}

		// Getting saved values
		$tab			= isset( $_POST[$prefix.'tab'] )			? paoc_pro_clean( $_POST[$prefix.'tab'] )			: '';
		$popup_goal		= isset( $_POST[$prefix.'popup_goal'] )		? paoc_pro_clean( $_POST[$prefix.'popup_goal'] )	: 'announcement';
		$display_type	= isset( $_POST[$prefix.'display_type'] )	? paoc_pro_clean( $_POST[$prefix.'display_type'] )	: 'modal';
		$popup_appear	= isset( $_POST[$prefix.'popup_appear'] )	? paoc_pro_clean( $_POST[$prefix.'popup_appear'] )	: 'page_load';
		
		// Behaviour Settings
		$behaviour							= isset( $_POST[$prefix.'behaviour'] )		? $_POST[$prefix.'behaviour']										: array();
		$behaviour['open_inactivity']		= isset( $behaviour['open_inactivity'] )	? paoc_pro_clean_number( $behaviour['open_inactivity'], '', 'abs')	: '';
		$behaviour['open_delay']			= isset( $behaviour['open_delay'] )			? paoc_pro_clean_number( $behaviour['open_delay'], '', 'abs')		: '';
		$behaviour['disappear']				= isset( $behaviour['disappear'] )			? paoc_pro_clean_number( $behaviour['disappear'], '', 'float')		: '';
		$behaviour['loader_speed']			= ! empty( $behaviour['loader_speed'] )		? paoc_pro_clean_number( $behaviour['loader_speed'], '', 'abs' )	: 1000;
		$behaviour['popup_img_id']			= ! empty( $behaviour['popup_img_id'] )		? paoc_pro_clean_number( $behaviour['popup_img_id'] )				: 0;
		$behaviour['open_scroll']			= isset( $behaviour['open_scroll'] )		? paoc_pro_clean_number( $behaviour['open_scroll'], 50) 			: 50;
		$behaviour['open_scroll']			= ( $behaviour['open_scroll'] <= 100 )		? $behaviour['open_scroll']											: 100;
		$behaviour['image_url']				= isset( $behaviour['image_url'] )			? paoc_pro_clean_url( $behaviour['image_url'] )						: '';
		$behaviour['btn_class']				= isset( $behaviour['btn_class'] )			? paoc_pro_sanitize_html_classes( $behaviour['btn_class'] )			: '';
		$behaviour['close_icon_tooltip']	= isset( $behaviour['close_icon_tooltip'] )	? paoc_pro_clean_html( $behaviour['close_icon_tooltip'] )			: '';
		$behaviour['btn_text']				= ! empty( $behaviour['btn_text'] )			? paoc_pro_clean_html( $behaviour['btn_text'] )						: '';
		$behaviour['btn_text']				= ! empty( $behaviour['btn_text'] )			? $behaviour['btn_text']											: __('Click Here!!!', 'popup-anything-on-click');
		$behaviour['link_text']				= ! empty( $behaviour['link_text'] )		? paoc_pro_clean_html( $behaviour['link_text'] )					: '';
		$behaviour['link_text']				= ! empty( $behaviour['link_text'] )		? $behaviour['link_text']											: __('Click Me!!!', 'popup-anything-on-click');
		$behaviour['image_title']			= ! empty( $behaviour['image_title'] )		? 1	: 0;
		$behaviour['image_caption']			= ! empty( $behaviour['image_caption'] )	? 1	: 0;
		$behaviour['hide_close']			= ! empty( $behaviour['hide_close'] )		? 1	: 0;
		$behaviour['clsonesc']				= ! empty( $behaviour['clsonesc'] )			? 1	: 0;
		$behaviour['enable_loader']			= ! empty( $behaviour['enable_loader'] )	? 1	: 0;
		$behaviour['close_overlay']			= ! empty( $behaviour['close_overlay'] )	? 1	: 0;
		$behaviour['hide_overlay']			= ! empty( $behaviour['hide_overlay'] )		? 1	: 0;
		$behaviour['close_overlay']			= ( $behaviour['hide_overlay'] == 1 )		? 0	: $behaviour['close_overlay'];

		// Content Settings
		$content						= isset( $_POST[$prefix.'content'] )		? $_POST[$prefix.'content']								: array();
		$content['main_heading']		= isset( $content['main_heading'] )			? paoc_pro_clean_html( $content['main_heading'] )		: '';
		$content['sub_heading']			= isset( $content['sub_heading'] )			? paoc_pro_clean_html( $content['sub_heading'] )		: '';
		$content['cust_close_txt']		= isset( $content['cust_close_txt'] )		? paoc_pro_clean( $content['cust_close_txt'] )			: '';
		$content['security_note']		= isset( $content['security_note'] )		? paoc_pro_clean( $content['security_note'] )			: '';
		$content['slide_in_text']		= ! empty( $content['slide_in_text'] )		? paoc_pro_clean_html( $content['slide_in_text'] )		: __('Click Here', 'popup-anything-on-click');
		$content['secondary_content']	= isset( $content['secondary_content'] )	? sanitize_post_field( 'post_excerpt', $content['secondary_content'], $post_id, 'db' )	: '';

		// Collect Lead Settings
		$form_fields				= array();
		$content['form_fields']		= isset( $content['form_fields'] )		? $content['form_fields']							: array();
		$content['submit_btn_txt']	= isset( $content['submit_btn_txt'] )	? paoc_pro_clean_html( $content['submit_btn_txt'] )	: '';
		$content['thanks_msg']		= isset( $content['thanks_msg'] )		? paoc_pro_clean_html( $content['thanks_msg'] )		: '';

		if( ! empty( $content['form_fields'] ) ) {
			foreach ($content['form_fields'] as $field_key => $field_data) {

				$field_require		= ! empty( $field_data['require'] )		? 1	: 0;
				$field_multiple		= ! empty( $field_data['multiple'] )	? 1	: 0;
				$field_type			= ! empty( $field_data['type'] )		? paoc_pro_clean( $field_data['type'] )				: 'text';
				$field_label		= isset( $field_data['label'] )			? paoc_pro_clean( $field_data['label'] )			: '';
				$field_placeholder	= isset( $field_data['placeholder'] )	? paoc_pro_clean( $field_data['placeholder'] )		: '';
				$field_choices		= isset( $field_data['choices'] )		? sanitize_textarea_field( $field_data['choices'] )	: '';
				$field_require		= ( $field_type == 'math_captcha' )		? 1 : $field_require;

				$field_choices_data	= '';
				$choices_arr		= explode( "\n", $field_choices );

				if( ! empty( $choices_arr ) ) {
					foreach ($choices_arr as $choices_key => $choices_data) {
						
						$choices_data	= explode( "|", $choices_data );
						$choices_val	= isset( $choices_data[0] )	? trim( $choices_data[0] )	: '';
						$choices_name	= isset( $choices_data[1] )	? trim( $choices_data[1] )	: '';
						$choices_name	= ! empty( $choices_name )	? $choices_name				: $choices_val;

						if( $choices_name != '' ) {
							$field_choices_data .= $choices_val ." | ". $choices_name . "\n";
						}
					}

					$field_choices_data = trim( $field_choices_data, "\n" );
				}

				$form_fields[ $field_key ]['type']			= $field_type;
				$form_fields[ $field_key ]['label']			= $field_label;
				$form_fields[ $field_key ]['placeholder']	= $field_placeholder;
				$form_fields[ $field_key ]['choices']		= $field_choices_data;
				$form_fields[ $field_key ]['multiple']		= $field_multiple;
				$form_fields[ $field_key ]['require']		= $field_require;
			}

			$content['form_fields'] = $form_fields;
		}

		// If empty of form fields (Incase if no form field is there)
		if( empty( $form_fields ) ) {
			$content['form_fields'][0] = array(
								'type'			=> __( 'email', 'popup-anything-on-click' ),
								'label'			=> __( 'Email', 'popup-anything-on-click' ),
								'placeholder'	=> __( 'Enter Your Email', 'popup-anything-on-click' ),
								'require'		=> 1,
							);
		}

		// Target URL Settings
		$content['target_url']['btn1_text']		= isset( $content['target_url']['btn1_text'] )		? paoc_pro_clean_html( $content['target_url']['btn1_text'] )	: '';
		$content['target_url']['btn1_link']		= isset( $content['target_url']['btn1_link'] )		? paoc_pro_clean_url( $content['target_url']['btn1_link'] )		: '';
		$content['target_url']['btn1_target']	= isset( $content['target_url']['btn1_target'] )	? paoc_pro_clean( $content['target_url']['btn1_target'] )		: '';
		$content['target_url']['btn2_text']		= isset( $content['target_url']['btn2_text'] )		? paoc_pro_clean_html( $content['target_url']['btn2_text'] )	: '';
		$content['target_url']['btn2_link']		= isset( $content['target_url']['btn2_link'] )		? paoc_pro_clean_url( $content['target_url']['btn2_link'] )		: '';
		$content['target_url']['btn2_target']	= isset( $content['target_url']['btn2_target'] )	? paoc_pro_clean( $content['target_url']['btn2_target'] )		: '';

		// Phone Calls Settings
		$content['phone_calls']['btn_txt']		= isset( $content['phone_calls']['btn_txt'] )	? paoc_pro_clean_html( $content['phone_calls']['btn_txt'] )			: '';
		$content['phone_calls']['phone_num']	= isset( $content['phone_calls']['phone_num'] )	? paoc_pro_clean_number( $content['phone_calls']['phone_num'], '' )	: '';

		// Social Settings
		$social_traffic_data		= array();
		$social						= isset( $_POST[$prefix.'social'] )		? $_POST[$prefix.'social']					: array();
		$social_type				= isset( $social['social_type'] )		? paoc_pro_clean( $social['social_type'] )	: '';
		$social['social_traffic']	= isset( $social['social_traffic'] )	? $social['social_traffic']					: array();

		if( ! empty( $social['social_traffic'] ) ) {
			foreach ($social['social_traffic'] as $social_key => $social_data) {

				$social_data_name = isset( $social_data['name'] ) ? paoc_pro_clean( $social_data['name'] ) : '';

				if( $social_data_name == 'whatsapp' || $social_data_name == 'skype' ) {

					$social_data_link = paoc_pro_clean( $social_data['link'] );

				} else if( $social_data_name == 'mail' ) {

					$social_data_link = paoc_pro_check_email( $social_data['link'] );

				} else {

					$social_data_link = paoc_pro_clean_url( $social_data['link'] );
				}

				if( empty( $social_data_name ) || empty( $social_data_link ) ) {
					continue;
				}

				$social_traffic_data[ $social_key ]['name'] = $social_data_name;
				$social_traffic_data[ $social_key ]['link'] = $social_data_link;
			}

			$social['social_traffic'] = $social_traffic_data;
		}

		// Design Settings
		$design							= isset( $_POST[$prefix.'design'] )			? $_POST[$prefix.'design']										: array();
		$design['bar_template']			= isset( $design['bar_template'] )			? paoc_pro_clean( $design['bar_template'] )						: '';
		$design['template']				= isset( $design['template'] )				? paoc_pro_clean( $design['template'] )							: '';
		$design['fullscreen_popup']		= ! empty( $design['fullscreen_popup'] )	? 1	: 0;
		$design['width']				= isset( $design['width'] )					? paoc_pro_clean( $design['width'] )							: '';
		$design['height']				= isset( $design['height'] )				? paoc_pro_clean_number( $design['height'], '' )				: '';
		$design['overlay_img']			= isset( $design['overlay_img'] )			? paoc_pro_clean_url( $design['overlay_img'] )					: '';
		$design['overlay_color']		= ! empty( $design['overlay_color'] )		? paoc_pro_clean_color( $design['overlay_color'] )				: 'rgba(11,11,11,0.8)';
		$design['speed_in']				= ! empty( $design['speed_in'] )			? paoc_pro_clean_number( $design['speed_in'], '', 'abs' )		: 0.5;
		$design['speed_out']			= ! empty( $design['speed_out'] )			? paoc_pro_clean_number( $design['speed_out'], '', 'abs' )		: 0.25;
		$design['popup_img']			= isset( $design['popup_img'] )				? paoc_pro_clean_url( $design['popup_img'] )					: '';
		$design['effect']				= isset( $design['effect'] )				? paoc_pro_clean( $design['effect'] )							: '';
		$design['loader_color']			= ! empty( $design['loader_color'] )		? paoc_pro_clean_color( $design['loader_color'] )				: '#000000';
		$design['bg_color']				= isset( $design['bg_color'] )				? paoc_pro_clean_color( $design['bg_color'] )					: '';
		$design['mheading_fontsize']	= isset( $design['mheading_fontsize'] )		? paoc_pro_clean_number( $design['mheading_fontsize'], '' )		: '';
		$design['mheading_txtcolor']	= isset( $design['mheading_txtcolor'] )		? paoc_pro_clean_color( $design['mheading_txtcolor'] )			: '';
		$design['sheading_fontsize']	= isset( $design['sheading_fontsize'] )		? paoc_pro_clean_number( $design['sheading_fontsize'], '' )		: '';
		$design['sheading_txtcolor']	= isset( $design['sheading_txtcolor'] )		? paoc_pro_clean_color( $design['sheading_txtcolor'] )			: '';
		$design['content_color']		= isset( $design['content_color'] )			? paoc_pro_clean_color( $design['content_color'] )				: '';
		$design['secondary_cnt_clr']	= isset( $design['secondary_cnt_clr'] )		? paoc_pro_clean_color( $design['secondary_cnt_clr'] )			: '';
		$design['custom_close_txtclr']	= isset( $design['custom_close_txtclr'] )	? paoc_pro_clean_color( $design['custom_close_txtclr'] )		: '';
		$design['snote_txtcolor']		= isset( $design['snote_txtcolor'] )		? paoc_pro_clean_color( $design['snote_txtcolor'] )				: '';

		// Target URL Settings
		$design['target_url']['btn1_bg_clr']	= isset( $design['target_url']['btn1_bg_clr'] )		? paoc_pro_clean_color( $design['target_url']['btn1_bg_clr'] )		: '';
		$design['target_url']['btn1_hbg_clr']	= isset( $design['target_url']['btn1_hbg_clr'] )	? paoc_pro_clean_color( $design['target_url']['btn1_hbg_clr'] )		: '';
		$design['target_url']['btn1_txt_clr']	= isset( $design['target_url']['btn1_txt_clr'] )	? paoc_pro_clean_color( $design['target_url']['btn1_txt_clr'] )		: '';
		$design['target_url']['btn1_htxt_clr']	= isset( $design['target_url']['btn1_htxt_clr'] )	? paoc_pro_clean_color( $design['target_url']['btn1_htxt_clr'] )	: '';
		$design['target_url']['btn2_bg_clr']	= isset( $design['target_url']['btn2_bg_clr'] )		? paoc_pro_clean_color( $design['target_url']['btn2_bg_clr'] )		: '';
		$design['target_url']['btn2_hbg_clr']	= isset( $design['target_url']['btn2_hbg_clr'] )	? paoc_pro_clean_color( $design['target_url']['btn2_hbg_clr'] )		: '';
		$design['target_url']['btn2_txt_clr']	= isset( $design['target_url']['btn2_txt_clr'] )	? paoc_pro_clean_color( $design['target_url']['btn2_txt_clr'] )		: '';
		$design['target_url']['btn2_htxt_clr']	= isset( $design['target_url']['btn2_htxt_clr'] )	? paoc_pro_clean_color( $design['target_url']['btn2_htxt_clr'] )	: '';

		// Collect Lead Settings
		$design['email_lists']['field_lbl_clr']	= isset( $design['email_lists']['field_lbl_clr'] )	? paoc_pro_clean_color( $design['email_lists']['field_lbl_clr'] )	: '';
		$design['email_lists']['field_bg_clr']	= isset( $design['email_lists']['field_bg_clr'] )	? paoc_pro_clean_color( $design['email_lists']['field_bg_clr'] )	: '';
		$design['email_lists']['field_txt_clr']	= isset( $design['email_lists']['field_txt_clr'] )	? paoc_pro_clean_color( $design['email_lists']['field_txt_clr'] )	: '';
		$design['email_lists']['btn_bg_clr']	= isset( $design['email_lists']['btn_bg_clr'] )		? paoc_pro_clean_color( $design['email_lists']['btn_bg_clr'] )		: '';
		$design['email_lists']['btn_hbg_clr']	= isset( $design['email_lists']['btn_hbg_clr'] )	? paoc_pro_clean_color( $design['email_lists']['btn_hbg_clr'] )		: '';
		$design['email_lists']['btn_txt_clr']	= isset( $design['email_lists']['btn_txt_clr'] )	? paoc_pro_clean_color( $design['email_lists']['btn_txt_clr'] )		: '';
		$design['email_lists']['btn_htxt_clr']	= isset( $design['email_lists']['btn_htxt_clr'] )	? paoc_pro_clean_color( $design['email_lists']['btn_htxt_clr'] )	: '';

		// Phone Calls Settings
		$design['phone_calls']['btn_bg_clr']	= isset( $design['phone_calls']['btn_bg_clr'] )		? paoc_pro_clean_color( $design['phone_calls']['btn_bg_clr'] )		: '';
		$design['phone_calls']['btn_hbg_clr']	= isset( $design['phone_calls']['btn_hbg_clr'] )	? paoc_pro_clean_color( $design['phone_calls']['btn_hbg_clr'] )		: '';
		$design['phone_calls']['btn_txt_clr']	= isset( $design['phone_calls']['btn_txt_clr'] )	? paoc_pro_clean_color( $design['phone_calls']['btn_txt_clr'] )		: '';
		$design['phone_calls']['btn_htxt_clr']	= isset( $design['phone_calls']['btn_htxt_clr'] )	? paoc_pro_clean_color( $design['phone_calls']['btn_htxt_clr'] )	: '';

		// Notification Settings
		$notification						= isset( $_POST[$prefix.'notification'] )			? $_POST[$prefix.'notification']							: array();
		$notification['email_to']			= ! empty( $notification['email_to'] )				? paoc_pro_check_email( $notification['email_to'] ) 		: get_option('admin_email');
		$notification['email_subject']		= ! empty( $notification['email_subject'] )			? paoc_pro_clean( $notification['email_subject'] ) 			: __('New submission received!', 'popup-anything-on-click');
		$notification['email_heading']		= ! empty( $notification['email_heading'] )			? paoc_pro_clean( $notification['email_heading'] ) 			: __('New Submission Received', 'popup-anything-on-click');
		$notification['email_msg']			= ! empty( $notification['email_msg'] )				? paoc_pro_clean_html( $notification['email_msg'] ) 		: __('Hi,<br/><br/>You have received a new submission. Form data is as follows.<br/><br/>{all_fields}', 'popup-anything-on-click');
		$notification['user_email_subject']	= ! empty( $notification['user_email_subject'] )	? paoc_pro_clean( $notification['user_email_subject'] ) 	: __('Hey! Thanks for your submission. Your entry has been received.', 'popup-anything-on-click');
		$notification['user_email_heading']	= ! empty( $notification['user_email_heading'] )	? paoc_pro_clean( $notification['user_email_heading'] ) 	: __('Thanks for your submission', 'popup-anything-on-click');
		$notification['user_email_msg']		= ! empty( $notification['user_email_msg'] )		? paoc_pro_clean_html( $notification['user_email_msg'] ) 	: __('Hi<br/><br/>Thanks for your submission. Your entry has been received. Form data is as follows.<br/><br/>{all_fields}', 'popup-anything-on-click');
		$notification['user_enable_email']	= ! empty( $notification['user_enable_email'] )		? 1	: 0;
		$notification['enable_email']		= ! empty( $notification['enable_email'] )			? 1	: 0;

		// Integration Settings
		$mc_intgs_data				= array();
		$skip_list_ids				= array();
		$integration				= isset( $_POST[$prefix.'integration'] )	? $_POST[$prefix.'integration']	: array();
		$integration['mc_enable']	= ! empty( $integration['mc_enable'] )		? 1	: 0;

		if( ! empty( $integration['mc_lists'] ) ) {
			foreach( $integration['mc_lists'] as $mc_list_key => $mc_list_data ) {

				$list_id = ! empty( $mc_list_data['list_id'] ) ? paoc_pro_clean( $mc_list_data['list_id'] ) : '';

				// Skip if no list id or same list id is there
				if ( empty( $list_id ) || in_array( $list_id, $skip_list_ids ) ) {
					continue;
				}

				$field_data_arr		= array();
				$mc_list_key		= paoc_pro_clean_number( $mc_list_key );
				$mc_tags			= isset( $mc_list_data['mc_tags'] )				? paoc_pro_clean( $mc_list_data['mc_tags'] )		: '';
				$mc_fields			= ! empty( $mc_list_data['mc_fields'] )			? paoc_pro_clean( $mc_list_data['mc_fields'] )		: array();
				$form_fields		= ! empty( $mc_list_data['form_fields'] )		? paoc_pro_clean( $mc_list_data['form_fields'] )	: array();
				$syn_mc_fields		= isset( $mc_list_data['syn_mc_fields'] )		? paoc_pro_clean( $mc_list_data['syn_mc_fields'] )	: array();
				$mc_double_optin	= ! empty( $mc_list_data['mc_double_optin'] )	? 1	: 0;

				foreach( $mc_fields as $mc_key => $mc_val ) {
					if( ! empty( $mc_val ) && isset( $form_fields[ $mc_key ] ) && $form_fields[ $mc_key ] != '' ) {
						$field_data_arr[ $mc_val ] = $form_fields[ $mc_key ];
					}
				}

				// If no field is mapped
				if ( empty( $field_data_arr ) ) {
					continue;
				}

				$mc_intgs_data[ $mc_list_key ]['list_id']			= $list_id;
				$mc_intgs_data[ $mc_list_key ]['mc_tags']			= $mc_tags;
				$mc_intgs_data[ $mc_list_key ]['mc_double_optin']	= $mc_double_optin;
				$mc_intgs_data[ $mc_list_key ]['syn_mc_fields']		= $syn_mc_fields;
				$mc_intgs_data[ $mc_list_key ]['fields']			= $field_data_arr;

				// Assign list ID to temp array
				$skip_list_ids[] = $list_id;
			}
		}
		$integration['mc_lists'] = $mc_intgs_data;

		// Advance Settings
		$advance					= isset( $_POST[$prefix.'advance'] )		? $_POST[$prefix.'advance']									: array();
		$advance['cookie_expire']	= isset( $advance['cookie_expire'] )		? paoc_pro_clean( $advance['cookie_expire'] ) 				: '';
		$advance['cookie_expire']	= ( $advance['cookie_expire'] != '' )		? paoc_pro_clean_number( $advance['cookie_expire'], null )	: '';
		$advance['cookie_unit']		= isset( $advance['cookie_unit'] )			? paoc_pro_clean( $advance['cookie_unit'] )					: 'days';
		$advance['show_for']		= isset( $advance['show_for'] )				? paoc_pro_clean( $advance['show_for'] )					: '';
		$advance['role']			= isset( $advance['role'] )					? paoc_pro_clean( $advance['role'] )						: array();
		$advance['display_on']		= isset( $advance['display_on'] )			? paoc_pro_clean( $advance['display_on'] )					: '';
		$advance['adblocker']		= ! empty( $advance['adblocker'] )			? paoc_pro_clean( $advance['adblocker'] )					: '';
		$advance['start_time']		= ! empty( $advance['start_time'] )			? paoc_pro_clean( $advance['start_time'] )					: '';
		$advance['end_time']		= ! empty( $advance['end_time'] )			? paoc_pro_clean( $advance['end_time'] ) 					: '';
		$advance['referrer_mode']	= ! empty( $advance['referrer_mode'] )		? paoc_pro_clean_number( $advance['referrer_mode'] )		: '';
		$advance['utm_mode']		= ! empty( $advance['utm_mode'] )			? paoc_pro_clean_number( $advance['utm_mode'] )				: '';
		$advance['utm_params']		= ! empty( $advance['utm_params'] )			? sanitize_textarea_field( $advance['utm_params'] )			: '';
		$advance['cookie_params']	= ! empty( $advance['cookie_params'] )		? sanitize_textarea_field( $advance['cookie_params'] )		: '';
		$advance['referrer_url']	= ! empty( $advance['referrer_url'] )		? $advance['referrer_url']									: '';
		$advance['referrer_enable']	= ! empty( $advance['referrer_enable'] )	? 1	: 0;
		$advance['utm_enable']		= ! empty( $advance['utm_enable'] )			? 1	: 0;
		$advance['cookie_enable']	= ! empty( $advance['cookie_enable'] )		? 1	: 0;
		$advance['show_credit']		= ! empty( $advance['show_credit'] )		? 1	: 0;
		$advance['store_no_views']	= ! empty( $advance['store_no_views'] )		? 1	: 0;
		$advance['store_no_data']	= ! empty( $advance['store_no_data'] )		? 1	: 0;

		// Schedule Popup Settings
		$advance['recur_days']			= isset( $advance['recur_days'] )			? paoc_pro_clean( $advance['recur_days'] )					: array();
		$advance['start_time']			= isset( $advance['start_time'] )			? paoc_pro_clean( $advance['start_time'] )					: '';
		$advance['end_time']			= isset( $advance['end_time'] )				? paoc_pro_clean( $advance['end_time'] )					: '';
		$advance['recur_start_time']	= isset( $advance['recur_start_time'] )		? paoc_pro_clean( $advance['recur_start_time'] )			: '';
		$advance['recur_end_time']		= isset( $advance['recur_end_time'] )		? paoc_pro_clean( $advance['recur_end_time'] )				: '';
		$advance['schedule_mode']		= ! empty( $advance['schedule_mode'] )		? paoc_pro_clean( $advance['schedule_mode'] )				: 'default';
		$advance['recurring_type']		= ! empty( $advance['recurring_type'] )		? paoc_pro_clean( $advance['recurring_type'] )				: 'daily';
		$advance['recur_week_start']	= ! empty( $advance['recur_week_start'] )	? paoc_pro_clean( $advance['recur_week_start'] )			: '1';
		$advance['recur_week_end']		= ! empty( $advance['recur_week_end'] )		? paoc_pro_clean( $advance['recur_week_end'] )				: '0';

		// Sanitize Referrer Parameters
		$ref_param_temp	= array();
		$ref_param_arr	= explode( "\n", $advance['referrer_url'] );

		if( ! empty( $ref_param_arr ) ) {
			foreach ( $ref_param_arr as $ref_param_key => $ref_param_val ) {

				$ref_param_val	= trim( $ref_param_val );

				if( empty( $ref_param_val ) ) {
					continue;
				}

				$ref_param_val	= untrailingslashit( urldecode( $ref_param_val ) );
				$referrer_data	= wp_parse_url( $ref_param_val );
				$url_host		= isset( $referrer_data['host'] ) ? $referrer_data['host'] : '';
				$url_path		= isset( $referrer_data['path'] ) ? $referrer_data['path'] : '';

				$ref_param_temp[ $ref_param_key ] = ( $url_host.''.$url_path );
			}

			$advance['referrer_url'] = implode("\n", $ref_param_temp);
		}

		// Sanitize UTM Parameters
		$utm_param_temp = array();
		$utm_param_arr	= explode( "\n", $advance['utm_params'] );

		if( ! empty( $utm_param_arr ) ) {
			foreach ($utm_param_arr as $utm_param_key => $utm_param_val) {

				$utm_par	= explode( '|', $utm_param_val );
				$utm_par[0]	= isset( $utm_par[0] ) ? sanitize_title( $utm_par[0] )	: '';
				$utm_par[1]	= isset( $utm_par[1] ) ? trim( $utm_par[1] )			: '';

				if( $utm_par[0] ) {
					$utm_param_temp[ $utm_param_key ] = $utm_par[0] .' | '. $utm_par[1];
				}
			}

			$advance['utm_params'] = implode("\n", $utm_param_temp);
		}

		// Sanitize Cookie Params
		$cookie_param_temp	= array();
		$cookie_param_arr	= explode( "\n", $advance['cookie_params'] );

		if( ! empty( $cookie_param_arr ) ) {
			foreach ($cookie_param_arr as $cookie_param_key => $cookie_param_val) {

				$cookie_par		= explode( '|', $cookie_param_val );
				$cookie_par[0]	= isset( $cookie_par[0] ) ? sanitize_title( $cookie_par[0] )	: '';
				$cookie_par[1]	= isset( $cookie_par[1] ) ? trim( $cookie_par[1] )				: '';

				if( $cookie_par[0] ) {
					$cookie_param_temp[ $cookie_param_key ] = $cookie_par[0] .' | '. $cookie_par[1];
				}
			}

			$advance['cookie_params'] = implode("\n", $cookie_param_temp);
		}

		// Geolocation Settings
		$advance['geo_countries'] = ! empty( $advance['geo_countries'] ) ? paoc_pro_clean( $advance['geo_countries'] ) : array();
		$advance['geo_countries'] = array_map( 'strtolower', $advance['geo_countries'] );

		// Custom CSS Settings
		$custom_css	= isset( $_POST[$prefix.'custom_css'] )	? sanitize_textarea_field( $_POST[$prefix.'custom_css'] ) : '';

		// Google Analytics Settings
		$analytics				= isset( $_POST[$prefix.'analytics'] )	? $_POST[$prefix.'analytics']				: array();
		$analytics['enable']	= ! empty( $analytics['enable'] )		? 1											: 0;
		$analytics['action']	= ! empty( $analytics['action'] )		? paoc_pro_clean( $analytics['action'] )	: __('Popup-Click', 'popup-anything-on-click');
		$analytics['category']	= ! empty( $analytics['category'] )		? paoc_pro_clean( $analytics['category'] )	: __('Popup-Open', 'popup-anything-on-click');
		$analytics['label']		= ! empty( $analytics['label'] )		? paoc_pro_clean( $analytics['label'] )		: __('Popup', 'popup-anything-on-click');

		// Update Meta
		update_post_meta( $post_id, $prefix.'tab', $tab );
		update_post_meta( $post_id, $prefix.'popup_goal', $popup_goal );
		update_post_meta( $post_id, $prefix.'display_type', $display_type );
		update_post_meta( $post_id, $prefix.'popup_appear', $popup_appear );
		update_post_meta( $post_id, $prefix.'behaviour', $behaviour );
		update_post_meta( $post_id, $prefix.'content', $content );
		update_post_meta( $post_id, $prefix.'design', $design );
		update_post_meta( $post_id, $prefix.'social', $social );
		update_post_meta( $post_id, $prefix.'notification', $notification );
		update_post_meta( $post_id, $prefix.'integration', $integration );
		update_post_meta( $post_id, $prefix.'advance', $advance );
		update_post_meta( $post_id, $prefix.'custom_css', $custom_css );
		update_post_meta( $post_id, $prefix.'analytics', $analytics );
	}

	/**
	 * Function to display DB updated notice
	 * 
	 * @since 2.0
	 */
	function paoc_pro_admin_notices() {

		// Database Update Notice
		if( isset($_GET['message']) && $_GET['message'] == 'paoc-db-update' ) {
			paoc_pro_display_message( 'update', __('Popup Anything Pro database update process has been done successfully.', 'popup-anything-on-click') );
		}
	}

	/**
	 * Function to remove popup id from campaign table on popup delete
	 * 
	 * @since 2.0
	 */
	function paoc_pro_delete_popup_data( $postid, $post ) {

		// Check for popup post type only
		if ( $post->post_type != PAOC_PRO_POST_TYPE ) {
			return;
		}

		global $wpdb;

		// Taking some variable
		$prefix			= PAOC_PRO_META_PREFIX;
		$campaign_id	= get_post_meta( $postid, $prefix.'campaign_id', true );

		// If campaign ID is there in popup meta
		if( ! empty( $campaign_id ) ) {

			// Taking some data
			$campaign_query = "SELECT * FROM ".PAOC_PRO_CAMPAIGN_TBL." WHERE 1=1 AND `id`={$campaign_id}";
			$campaign_data	= $wpdb->get_row( $campaign_query );
			$popup_ids		= isset( $campaign_data->popup_ids ) ? explode( ',', $campaign_data->popup_ids ) : array();

			if ( empty( $popup_ids ) ) {
				return;
			}

			// Remove popup id from campaign
			$popup_key = array_search( $postid, $popup_ids );

			if ( $popup_key === FALSE ) {
				return;
			}

			unset( $popup_ids[ $popup_key ] );

			$popup_ids = implode( ',', $popup_ids );

			// Update Campaign Data
			$wpdb->update( PAOC_PRO_CAMPAIGN_TBL,
							array(
								'popup_ids'		=> $popup_ids,
								'modified_date'	=> current_time('mysql'),
							),
							array(
								'id' => $campaign_id,
							)
						);
		}
	}

	/**
	 * Function register setings
	 * 
	 * @package Popup Anything on Click
	 * @since 1.2.4
	 */
	function paoc_pro_admin_processes() {

		// If plugin notice is dismissed
		if( isset($_GET['message']) && $_GET['message'] == 'paoc-pro-plugin-notice' ) {
			set_transient( 'paoc_pro_install_notice', true, 604800 );
		}

		// If plugin notice is dismissed
		if( isset($_GET['message']) && $_GET['message'] == 'paoc-pro-license-exp-notice' ) {
			set_transient( 'paoc_pro_license_exp_notice', true, 864000 );
		}
	}

	/**
	 * Add custom column to Post listing page
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.0
	 */
	function paoc_pro_manage_posts_columns( $columns ) {

		$new_columns['paoc_popup_goal']		= esc_html__('Goal', 'popup-anything-on-click');
		$new_columns['paoc_display_type']	= esc_html__('Type', 'popup-anything-on-click');
		$new_columns['paoc_popup_appear']	= esc_html__('Appear On', 'popup-anything-on-click');
		$new_columns['paoc_popup_view']		= esc_html__('Impression', 'popup-anything-on-click');
		$new_columns['paoc_popup_click']	= esc_html__('Clicks', 'popup-anything-on-click');
		$new_columns['paoc_quick_overview']	= esc_html__('Quick Overview', 'popup-anything-on-click');

		$columns = paoc_pro_add_array( $columns, $new_columns, 1, true );

		return $columns;
	}

	/**
	 * Add custom column data to Post listing page
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.0
	 */
	function paoc_pro_post_columns_data( $column, $post_id ) {

		$prefix 		= PAOC_PRO_META_PREFIX;
		$popup_views 	= 0;
		$popup_clicks 	= 0;

		switch ($column) {
			case 'paoc_popup_goal':
				$popup_goals	= paoc_pro_popup_goals();
				$popup_goal		= get_post_meta( $post_id, $prefix.'popup_goal', true );
				$popup_goal		= isset( $popup_goals[ $popup_goal ]['name'] ) ? $popup_goals[ $popup_goal ]['name'] : $popup_goal;

				echo esc_html( $popup_goal );
				break;

			case 'paoc_display_type':
				$popup_types	= paoc_pro_popup_types();
				$display_type	= get_post_meta( $post_id, $prefix.'display_type', true );
				$display_type	= isset( $popup_types[ $display_type ]['name'] ) ? $popup_types[ $display_type ]['name'] : $display_type;

				echo esc_html( $display_type );
				break;

			case 'paoc_popup_appear':
				$appear_types	= paoc_pro_when_appear_options();
				$popup_appear	= get_post_meta( $post_id, $prefix.'popup_appear', true );
				$popup_appear	= isset( $appear_types[ $popup_appear ] ) ? $appear_types[ $popup_appear ] : $popup_appear;
				
				echo esc_html( $popup_appear );
				break;

			case 'paoc_popup_view':
				$popup_views			= get_post_meta( $post_id, $prefix.'popup_views', true );
				$popup_views_normal		= ! empty( $popup_views['normal'] ) ? $popup_views['normal'] : 0;
				$popup_views_inline		= ! empty( $popup_views['inline'] ) ? $popup_views['inline'] : 0;
				$total_views			= ( $popup_views_normal + $popup_views_inline );
				$popup_view_tooltip		= sprintf( esc_html__("Normal Impression(s) : %d\nInline Impression(s) : %d", 'popup-anything-on-click'), $popup_views_normal, $popup_views_inline );

				echo '<abbr title="'.esc_attr( $popup_view_tooltip ).'">'. number_format( $total_views ) .'</abbr>';
				break;

			case 'paoc_popup_click':
				$popup_clicks			= get_post_meta( $post_id, $prefix.'popup_clicks', true );
				$popup_clicks_normal	= ! empty( $popup_clicks['normal'] ) ? $popup_clicks['normal'] : 0;
				$popup_clicks_inline	= ! empty( $popup_clicks['inline'] ) ? $popup_clicks['inline'] : 0;
				$total_clicks			= ( $popup_clicks_normal + $popup_clicks_inline );
				$popup_click_tooltip	= sprintf( esc_html__("Normal Click(s) : %d\nInline Click(s) : %d", 'popup-anything-on-click'), $popup_clicks_normal, $popup_clicks_inline );

				echo '<abbr title="'.esc_attr( $popup_click_tooltip ).'">'. number_format( $total_clicks ) .'</abbr>';
				break;

			case 'paoc_quick_overview':

				echo '<button type="button" class="paoc-dashicons-icon paoc-open-ajax-popup" title="'.esc_attr( get_the_title( $post_id ) ).' - '.esc_html__('Quick Overview', 'popup-anything-on-click').'" data-popup-id="'.esc_attr( $post_id ).'" data-nonce="'.wp_create_nonce( 'paoc-quick-overview-'.$post_id ).'" data-id="quick_overview">
							<span class="paoc-inline-loader"><i class="dashicons dashicons-update"></i></span>
							<i class="paoc-btn-txt-icon dashicons dashicons-visibility"></i>
					  </button>';
				break;
		}
	}

	/**
	 * Function to add duplicate post link at edit post sidebar
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 2.0
	*/
	function paoc_pro_duplicate_post_link() {

		global $post;

		$duplicate_link = add_query_arg( array( 'post_type' => PAOC_PRO_POST_TYPE, 'action' => 'paoc_pro_duplicate_post', 'paoc_post' => $post->ID, 'nonce' => wp_create_nonce( 'paoc-duplicate-post-'.$post->ID ) ), admin_url('admin.php') );

		if( $post->post_type == PAOC_PRO_POST_TYPE ) {

			echo '<div class="misc-pub-section paoc-duplicate-link-wrap">
					<i class="dashicons dashicons-admin-page"></i> <a href="'.esc_url( $duplicate_link ).'" title="'.esc_attr__('Duplicate this popup as draft', 'popup-anything-on-click').'" class="paoc-confirm paoc-duplicate-post">'.esc_html__('Duplicate this Popup', 'popup-anything-on-click').'</a>
				  </div>';
		}
	}

	/**
	 * Function to add custom quick links at post listing page
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 2.0
	*/
	function paoc_pro_post_row_action( $actions, $post ) {

		if( $post->post_type == PAOC_PRO_POST_TYPE ) {

			if ( current_user_can( 'manage_options' ) ) {
				$entry_link					= add_query_arg( array( 'post_type' => PAOC_PRO_POST_TYPE, 'page' => 'paoc-form-entries', 'paoc_popup_id' => $post->ID ), admin_url( 'edit.php' ) );
				$new_actions['paoc_entry']	= '<a href="'.esc_url( $entry_link ).'" target="_blank">'.esc_html__( 'Entries', 'popup-anything-on-click' ).'</a>';
			}

			if ( current_user_can( 'edit_post', $post->ID ) ) {
				$duplicate_link	= add_query_arg( array( 'post_type' => PAOC_PRO_POST_TYPE, 'action' => 'paoc_pro_duplicate_post', 'paoc_post' => $post->ID, 'nonce' => wp_create_nonce( 'paoc-duplicate-post-'.$post->ID ) ), admin_url('admin.php') );
				$new_actions['paoc_duplicate'] = '<a href="'.$duplicate_link.'" title="'.esc_html__('Duplicate this popup as draft', 'popup-anything-on-click').'" class="paoc-duplicate-post">'.esc_attr__('Duplicate', 'popup-anything-on-click').'</a>';
			}

			if ( ! empty( $new_actions ) ) {
				$actions = paoc_pro_add_array( $actions, $new_actions, 2 );
			}
		}

		return $actions;
	}

	/**
	 * Function to add extra link to plugins action link
	 * 
	 * @package Popup Anything on Click Pro
	 * @since 1.0
	 */
	function paoc_pro_plugin_row_meta( $links, $file ) {

		if ( $file == PAOC_PRO_PLUGIN_BASENAME ) {

			$row_meta = array(
				'docs'		=> '<a href="' . esc_url('https://docs.essentialplugin.com/popup-anything-on-click-pro/?utm_source=popup_pro&utm_medium=plugin_list&utm_campaign=plugin_quick_link') . '" title="' . esc_html__( 'View Documentation', 'popup-anything-on-click' ) . '" target="_blank">' . esc_html__( 'Docs', 'popup-anything-on-click' ) . '</a>',
				'support'	=> '<a href="' . esc_url('https://www.wponlinesupport.com/wordpress-services/?utm_source=popup_pro&utm_medium=plugin_list&utm_campaign=plugin_quick_link') . '" title="' . esc_html__( 'Premium Support - For any Customization', 'popup-anything-on-click' ) . '" target="_blank">' . esc_html__( 'Premium Support', 'popup-anything-on-click' ) . '</a>',
			);
			return array_merge( $links, $row_meta );
		}
		return (array) $links;
	}

	/**
	 * Function to Duplicate the post
	 * 
 	 * @since 2.0
	 */
	function paoc_pro_duplicate_post() {

		global $wpdb;

		// Taking some variale
		$prefix		= PAOC_PRO_META_PREFIX;
		$suffix		= ' - ' . esc_html__('Duplicate', 'popup-anything-on-click');
		$nonce		= isset( $_GET['nonce'] )		? $_GET['nonce']								: '';
		$post_id	= isset( $_GET['paoc_post'] )	? paoc_pro_clean_number( $_GET['paoc_post'] )	: 0;

		// Check `Popup ID` & `Nonce`
		if( $post_id && wp_verify_nonce( $nonce, 'paoc-duplicate-post-'.$post_id ) && current_user_can('edit_posts') ) {

			// Taking some variable
			$post_data = get_post( $post_id );

			// If check post data is there
			if( ! empty( $post_data->post_type ) && $post_data->post_type == PAOC_PRO_POST_TYPE ) {

				$popup_args = array(
					'post_type'			=> $post_data->post_type,
					'post_status'		=> 'draft',
					'post_title'		=> $post_data->post_title . $suffix,
					'post_content'		=> $post_data->post_content,
					'post_excerpt'		=> $post_data->post_excerpt,
					'post_author'		=> $post_data->post_author,
					'post_parent'		=> $post_data->post_parent,
					'post_password'		=> $post_data->post_password,
					'to_ping'			=> $post_data->to_ping,
					'ping_status'		=> $post_data->ping_status,
					'comment_status'	=> $post_data->comment_status,
					'menu_order'		=> $post_data->menu_order,
				);

				// Insert Post
				$inserted_post_id = wp_insert_post( $popup_args );

				// Get all post meta's
				$post_meta_data	= get_post_custom( $post_id );

				// Filter to skip the some meta key
				$skip_meta_keys = apply_filters( 'paoc_pro_skip_meta_keys', array(
																				'_edit_lock',
																				$prefix.'popup_views',
																				$prefix.'popup_clicks',
																				$prefix.'stats_flush_date',
																				$prefix.'campaign_enable',
																				$prefix.'campaign_id',
																			) );

				if( ! empty( $post_meta_data ) ) {
					foreach($post_meta_data as $post_meta_key => $post_meta_val) {

						if( in_array( $post_meta_key, $skip_meta_keys ) ) {
							continue;
						}

						$update_val = $post_meta_val[0];
						$update_val = maybe_unserialize( $update_val );

						update_post_meta( $inserted_post_id, $post_meta_key, $update_val );
					}

					// Redirect to popup listing page after the duplicate post
					$redirect_link	= add_query_arg( array( 'post_type' => PAOC_PRO_POST_TYPE, 'orderby' => 'date', 'order' => 'desc' ), admin_url('edit.php') );
					wp_redirect( $redirect_link );
					exit;
				}

			} else {

				wp_die( __('Sorry, Something happened wrong. Popup Anything on Click can not create duplicate popup.', 'popup-anything-on-click'), null, array('back_link' => true) );
			}
		} else {

			wp_die( __('Sorry, Something happened wrong. Popup Anything on Click can not create duplicate popup.', 'popup-anything-on-click'), null, array('back_link' => true) );
		}
	}

	/**
	 * Function to get post suggestion based on search input
	 * 
 	 * @since 2.0
	 */
	function paoc_pro_post_title_sugg() {

		$return			= array();
		$prefix			= PAOC_PRO_META_PREFIX;
		$post_status	= ! empty( $_GET['post_status'] )	? $_GET['post_status']							: 'publish';
		$search			= isset( $_GET['search'] )			? trim( $_GET['search'] )						: '';
		$post_type		= isset( $_GET['post_type'] )		? $_GET['post_type']							: 'post';
		$nonce			= isset( $_GET['nonce'] )			? $_GET['nonce']								: '';
		$post_data		= isset( $_GET['form_data'] )		? parse_str( $_GET['form_data'], $form_data )	: '';
		$meta_data		= isset( $_GET['meta_data'] )		? paoc_pro_clean( $_GET['meta_data'] )			: '';
		$meta_data		= json_decode( $meta_data, true );

		// Verify Nonce
		if( $search && wp_verify_nonce( $nonce, 'paoc-post-title-sugg' ) ) {

			$args = array(
						's'						=> $search,
						'post_type'				=> $post_type,
						'post_status'			=> $post_status,
						'order'					=> 'ASC',
						'orderby'				=> 'title',
						'posts_per_page'		=> 20
					);

			// If number is passed
			if( is_numeric( $search ) ) {
				$args['s'] = false;
				$args['p'] = $search;
			}

			// If meta query is set
			if( $meta_data ) {
				$args['meta_query'] = $meta_data;
			}

			// Form data is there
			if( ! empty( $form_data['popup_appear'] ) ) {
				$args['meta_query'][] = array(
											'key'		=> $prefix.'popup_appear',
											'value'		=> $form_data['popup_appear'],
											'compare'	=> '=',
										);
			}

			$search_query = get_posts( $args );

			if( $search_query ) :

				foreach ( $search_query as $search_data ) {
					
					$post_title	= ! empty( $search_data->post_title ) ? $search_data->post_title : esc_html__('Post', 'popup-anything-on-click');
					$post_title	= $post_title . " - (#{$search_data->ID})";

					$return[]	= array( $search_data->ID, $post_title );
				}

			endif;
		}

		wp_send_json( $return );
	}

	/**
	 * Function to flush the popup stats
	 * 
	 * @since 2.0
	 */
	function paoc_pro_flush_popup_stats() {

		$prefix		= PAOC_PRO_META_PREFIX;
		$result		= array( 'success' => 0 );
		$nonce		= isset( $_POST['nonce'] )		? paoc_pro_clean( $_POST['nonce'] )				: '';
		$popup_id	= isset( $_POST['popup_id'] )	? paoc_pro_clean_number( $_POST['popup_id'] )	: 0;

		if( $popup_id && wp_verify_nonce( $nonce, 'paoc-flush-stats-'.$popup_id ) ) {

			update_post_meta( $popup_id, $prefix.'popup_clicks', '' );
			update_post_meta( $popup_id, $prefix.'popup_views', '' );
			update_post_meta( $popup_id, $prefix.'stats_flush_date', current_time('mysql') );

			$result = array( 'success' => 1 );
		}

		wp_send_json( $result );
	}
}

$paoc_pro_admin = new Paoc_Pro_Admin();