<?php
/**
 * Form Entries HTML
 *
 * @package Popup Anything on Click Pro
 * @since 2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if( ! class_exists( 'WP_List_Table' ) ) {
	require_once( ABSPATH . 'wp-admin/includes/class-wp-list-table.php' );
}

class Paoc_Pro_Entries_List extends WP_List_Table {

	var $prefix, $per_page, $entries_count, $redirect_url, $popup_post_data, $curr_popup_id;

	// Construct
	function __construct() {

		$this->prefix       	= PAOC_PRO_META_PREFIX;
		$this->per_page    	 	= apply_filters( 'paoc_pro_form_entries_per_page', 15 ); // Per page
		$this->redirect_url		= add_query_arg( array('post_type' => PAOC_PRO_POST_TYPE, 'page' => 'paoc-form-entries'), admin_url('edit.php') );
		$this->popup_post_data	= paoc_pro_get_posts();

		$popup_id				= ! empty( $_GET['paoc_popup_id'] )	? $_GET['paoc_popup_id'] : '';
		$this->curr_popup_id	= ( empty( $popup_id ) && isset( $this->popup_post_data[0]->ID ) ) ? $this->popup_post_data[0]->ID : $popup_id;

		// Set parent defaults
		parent::__construct( array(
								'singular'  => 'paoc_pro_entry',	// singular name of the listed records
								'plural'    => 'paoc_pro_entry',	// plural name of the listed records
								'ajax'      => true					// does this table support ajax?
							));
	}

	/**
	 * Displaying emails data
	 * 
	 * Does prepare the data for displaying the emails in the table.
	 * 
	 * @since 2.0
	 */
	function paoc_pro_form_entries() {

		global $wpdb;

		// Taking some variable
		$entries	= array();
		$orderby	= isset( $_GET['orderby'] ) ? urldecode( $_GET['orderby'] ) : 'created_date';
		$order		= isset( $_GET['order'] )   ? $_GET['order']                : 'DESC';
		$args		= array(
						'limit'		=> $this->per_page,
						'popup_id'	=> $this->curr_popup_id,
						'order'		=> $order,
						'orderby'	=> $orderby,
						'search'	=> isset( $_GET['s'] ) ? $_GET['s'] : '',
					);

		// Get Emails Data
		$entries_data_arr		= paoc_pro_get_entries( $args );
		$entries_data			= $entries_data_arr['data'];
		$this->entries_count	= $entries_data_arr['count'];

		if( ! empty( $entries_data ) ) {
			foreach ( $entries_data as $entry_key => $entry_data ) {

				// Taking some variable
				$popup_id		= isset( $entry_data['popup_id'] ) ? $entry_data['popup_id'] : 0;
				$content		= get_post_meta( $popup_id, $this->prefix.'content', true );
				$form_fields	= isset( $content['form_fields'] ) ? $content['form_fields'] : array();

				$entries[$entry_key]['id']				= $entry_data['id'];
				$entries[$entry_key]['popup_id']		= $popup_id;
				$entries[$entry_key]['created_date']	= $entry_data['created_date'];

				foreach( $form_fields as $field_key => $field_data ) {
					$entries[ $entry_key ]['meta_key_'.$field_key] = isset( $entry_data['meta_key_'.$field_key] )	? $entry_data['meta_key_'.$field_key]	: '';
				}
			}
		}

		return $entries;
	}

	/**
	 * Display Columns
	 * 
	 * Handles which columns to show in table
	 * 
	 * @since 2.0
	 */
	function get_columns() {

		// Taking some data
		$count			= 0;
		$content		= get_post_meta( $this->curr_popup_id, $this->prefix.'content', true );
		$form_fields	= isset( $content['form_fields'] ) ? $content['form_fields'] : array();

		$columns = array(
			'cb'	=> '<input type="checkbox" />', // Render a checkbox instead of text
			'id'	=> esc_html__('ID', 'popup-anything-on-click'),
		);

		// Loop of form fields - Take only three fields
		foreach( $form_fields as $field_key => $field_data ) {

			if( $field_data['type'] == 'math_captcha' ) {
				continue;
			}

			$count++;
			$field_data['label']				= ( $field_data['label'] !== '' ) ? $field_data['label'] : esc_html__('Field', 'popup-anything-on-click') .' - '. $field_key;
			$columns['meta_key_'.$field_key]	= $field_data['label'];

			if( $count >= 3 ) {
				break;
			}
		}

		$columns['popup_id']		= esc_html__('Popup', 'popup-anything-on-click');
		$columns['created_date']	= esc_html__('Created Date', 'popup-anything-on-click');

		return $columns;
	}

	/**
	 * Mange column data
	 * 
	 * Default Column for listing table
	 * 
	 * @since 2.0
	 */
	function column_default( $item, $column_name ) {

		switch( $column_name ) {
			case "popup_id":
				$popup_id 		= $item['popup_id'];
				$default_val	= '<a href="'. esc_url( get_edit_post_link( $popup_id ) ) .'" target="_self" title="'.esc_attr__('Click to edit Popup', 'popup-anything-on-click').'">'. esc_html( get_the_title( $popup_id ) ) .'</a>';
				break;

			default:
				$default_val = ( $item[ $column_name ] !== '' ) ? wp_kses_post( $item[ $column_name ] ) : "&mdash;";
				break;
		}
		return $default_val;
	}

	/**
	 * Handles checkbox HTML
	 * 
	 * @since 2.0
	 **/
	function column_cb( $item ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			$this->_args['singular'],  // Let's simply repurpose the table's singular label ("paoc-emails")
			$item['id']                // The value of the checkbox should be the record's id
		);
	}

	/**
	 * Sortable Columns
	 *
	 * Handles soratable columns of the table
	 * 
	 * @since 2.0
	 */
	function get_sortable_columns() {

		$sortable_columns = array(
			'id'    		=> array('id', false),
			'created_date'  => array('created_date', false)
		);
		return $sortable_columns;
	}

	/**
	 * Manage Entries Column
	 *
	 * @since 2.0
	 */
	function column_id( $item ) {

		$paged 				= isset( $_GET['paged'] )			? $_GET['paged']			: false;
		$paoc_popup_id 		= isset( $_GET['paoc_popup_id'] )	? $_GET['paoc_popup_id']	: false;
		$view_entry 		= add_query_arg( array( 'action' => 'view', 'entry_id' => $item['id'], 'paoc_popup_id' => $paoc_popup_id ), $this->redirect_url );
		$delete_entries 	= add_query_arg( array( 'paged' => $paged, 'paoc_popup_id' => $paoc_popup_id, 'action' => 'delete', 'paoc_pro_entry[]' => $item['id'], '_wpnonce' => wp_create_nonce('bulk-paoc_pro_entry') ), $this->redirect_url );

		$actions['view']	= sprintf( '<a class="paoc-action-link" href="%s">'.esc_html__('View', 'popup-anything-on-click').'</a>', esc_url( $view_entry ) );
		$actions['delete']	= sprintf( '<a class="paoc-confirm paoc-action-link" href="%s">'.esc_html__('Delete', 'popup-anything-on-click').'</a>', esc_url( $delete_entries ) );

		// Return contents
		return sprintf('%1$s %2$s',
		   /*%1$s*/ $item['id'],
		   /*%2$s*/ $this->row_actions( $actions )
		);
	}

	/**
	 * Bulk actions field
	 *
	 * Handles Bulk Action combo box values
	 * 
	 * @since 2.0
	 */
	function get_bulk_actions() {
		$actions = array(
						'delete' => esc_html__('Delete', 'popup-anything-on-click')
					);
		return $actions;
	}

	/**
	 * Message to show when no records in database table
	 *
	 * @since 2.0
	 */
	function no_items() {
		echo esc_html__('No entries found.', 'popup-anything-on-click');
	}

	/**
	 * Add Filter for Popup dropdown
	 * Handles to add filter for sorting add popup selection dropdown
	 * 
	 * @since 1.0
	 **/
	function extra_tablenav( $which ) {

		if( $which == 'top' ) {

			$html = '';
			$html .= '<div class="alignleft actions paoc-popup-id-actions">';

			$html .= '<select name="paoc_popup_id" class="paoc-select-popup-id">';

			if( ! empty( $this->popup_post_data ) ) {
				foreach ( $this->popup_post_data as $popup_post_key => $popup_post_val ) {
					$html .= '<option value="'.esc_attr( $popup_post_val->ID ).'" '. selected( $popup_post_val->ID, $this->curr_popup_id, false ) .'>'. esc_html( $popup_post_val->post_title ) .'</option>';
				}
			}
			
			$html .= '</select><!-- end .paoc-select-popup-id -->';
			
			$html .= '<input type="submit" value="'.esc_html__( 'Filter', 'popup-anything-on-click' ).'" class="button paoc-popup-filters" id="paoc-popup-filters" name="paoc_popup_filters">';
			
			$html .= '</div><!-- end .paoc-popup-id-actions -->';
			
			echo $html; // phpcs:ignore WordPress.XSS.EscapeOutput.OutputNotEscaped
		}
	}

	/**
	 * Prepare Items for emails listing
	 * 
	 * @since 2.0
	 **/
	function prepare_items() {

		// Get how many records per page to show
		$per_page	= $this->per_page;

		// Get All, Hidden, Sortable columns
		$hidden		= array();
		$columns	= $this->get_columns();
		$sortable	= $this->get_sortable_columns();

		// Get final column header
		$this->_column_headers = array($columns, $hidden, $sortable);

		// Get current page number
		$current_page = $this->get_pagenum();

		// Get page items
		$this->items = $this->paoc_pro_form_entries();

		// Get total count
		$total_items = $this->entries_count;

		// Register pagination options and calculations.
		$this->set_pagination_args( array(
										'total_items' => $total_items,                  // Calculate the total number of items
										'per_page'    => $per_page,                     // Determine how many items to show on a page
										'total_pages' => ceil($total_items / $per_page)	// Calculate the total number of pages
									));
	}
}

$entries_list = new Paoc_Pro_Entries_List();
$entries_list->prepare_items();

// Little Patch to avoid too long URL Error
paoc_pro_avoid_long_url();
?>

<div class="wrap paoc-form-entries-wrap">

	<h2><?php esc_html_e( 'Popup Anything Pro - Form Entries', 'popup-anything-on-click' ); ?></h2>

	<?php if( ! empty($_GET['message']) && $_GET['message'] == 1 ) {
		paoc_pro_display_message( 'update', __('Entry record(s) deleted successfully.', 'popup-anything-on-click') );
	} ?>

	<form id="paoc-from-entries" class="paoc-from-entries" method="get" action="">

		<input type="hidden" name="post_type" value="<?php echo esc_attr( $_REQUEST['post_type'] ); ?>" />
		<input type="hidden" name="page" value="<?php echo esc_attr( $_REQUEST['page'] ); ?>" />

		<?php
			$entries_list->search_box( esc_html__( 'Search', 'popup-anything-on-click' ), 'popup-anything-on-click' );

			$entries_list->views();	// Showing sorting links on the top of the list

			$entries_list->display(); // Display all the entries
		?>
	</form><!-- end .paoc-from-entries -->
</div><!-- end .paoc-form-entries-wrap -->