<?php
/**
 * Register Settings
 *
 * Handles the Admin side setting options functionality of module
 *
 * @package Popup Anything on Click Pro
 * @since 2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Get settings tab
 * 
 * @since 2.0
 */
function paoc_pro_settings_tab() {

	// Plugin settings tab
	$sett_tabs = array(
					'general'		=> __( 'General', 'popup-anything-on-click' ),
					'display_rule'	=> __( 'Display Rule', 'popup-anything-on-click' ),
					'integration'	=> __( 'Integration', 'popup-anything-on-click' ),
				);

	return apply_filters( 'paoc_pro_settings_tab', (array)$sett_tabs );
}

/**
 * Set Settings Default Option Page
 * Handles to return all settings value
 *
 * @package Popup Anything On Click Pro
 * @since 1.3
 */
function paoc_pro_default_settings() {

	global $paoc_pro_options;

	$default_options = array(
						'add_js'				=> '',
						'enable'				=> 1,
						'geo_enable'			=> 1,
						'cookie_prefix'			=> 'paoc_popup',
						'post_types'			=> array('post'),
						'welcome_popup'			=> '',
						'welcome_display_in'	=> array(),
						'exit_popup'			=> '',
						'exit_display_in'		=> array(),
						'general_popup'			=> array(),
						'general_display_in'	=> array(),
						'mc_api_key'			=> '',
					);

	$default_options = apply_filters('paoc_pro_default_settings', $default_options );

	// Update default options
	update_option( 'paoc_pro_options', $default_options );

	// Overwrite global variable when option is update
	$paoc_pro_options = paoc_pro_get_settings();
}

/**
 * Function to register plugin settings
 * 
 * @since 2.0
 */
function paoc_pro_register_settings() {

	// Reset default settings
	if( ! empty( $_POST['paoc_pro_reset_settings'] ) && check_admin_referer( 'paoc_reset_setting', 'paoc_reset_sett_nonce' ) ) {
		
		// Default Settings
		paoc_pro_default_settings();
		
		// Integration Data
		update_option( 'paoc_intgs_info', '', false );
	}

	register_setting( 'paoc_pro_plugin_options', 'paoc_pro_options', 'paoc_pro_validate_options' );
}

// Action to register plugin settings
add_action( 'admin_init', 'paoc_pro_register_settings' );

/**
 * Validate Settings Options
 * 
 * @since 2.0
 */
function paoc_pro_validate_options( $input ) {

	global $paoc_pro_options;

	$input = $input ? $input : array();

	// Pull out the tab and section
	if ( isset ( $_POST['_wp_http_referer'] ) ) {
		parse_str( $_POST['_wp_http_referer'], $referrer );
	}

	$tab = isset( $referrer['tab'] ) ? paoc_pro_clean( $referrer['tab'] ) : 'general';

	// Run a general sanitization for the tab for special fields
	$input = apply_filters( 'paoc_pro_sett_sanitize_'.$tab, $input );

	// Run a general sanitization for the custom created tab
	$input = apply_filters( 'paoc_pro_sett_sanitize', $input, $tab );

	// Making merge of old and new input values
	$input = array_merge( $paoc_pro_options, $input );

	return $input;
}

/**
 * Filter to validate general settings
 * 
 * @since 2.0
 */
function paoc_pro_sanitize_general_sett( $input ) {

	$input['enable']		= ! empty( $input['enable'] )			? 1											: 0;
	$input['geo_enable']	= ! empty( $input['geo_enable'] )		? 1											: 0;
	$input['cookie_prefix']	= ! empty( $input['cookie_prefix'] )	? sanitize_title( $input['cookie_prefix'] )	: 'paoc_popup';
	$input['add_js']		= ! empty( $input['add_js'] )			? paoc_pro_clean_number( $input['add_js'] )	: '';

	return $input;
}
add_filter( 'paoc_pro_sett_sanitize_general', 'paoc_pro_sanitize_general_sett' );

/**
 * Filter to validate display rule settings
 * 
 * @since 2.0
 */
function paoc_pro_sanitize_display_rule_sett( $input ) {

	$input['post_types']			= ! empty( $input['post_types'] )			? paoc_pro_clean( $input['post_types'] )			: array();
	$input['welcome_popup']			= ! empty( $input['welcome_popup'] )		? paoc_pro_clean_number( $input['welcome_popup'] )	: '';
	$input['welcome_display_in']	= ! empty( $input['welcome_display_in'] )	? paoc_pro_clean( $input['welcome_display_in'] )	: array();
	$input['exit_popup']			= ! empty( $input['exit_popup'] )			? paoc_pro_clean_number( $input['exit_popup'] )		: '';
	$input['exit_display_in']		= ! empty( $input['exit_display_in'] )		? paoc_pro_clean( $input['exit_display_in'] )		: array();
	$input['general_popup']			= ! empty( $input['general_popup'] )		? paoc_pro_clean( $input['general_popup'] )			: array();
	$input['general_display_in']	= ! empty( $input['general_display_in'] )	? paoc_pro_clean( $input['general_display_in'] )	: array();

	return $input;
}
add_filter( 'paoc_pro_sett_sanitize_display_rule', 'paoc_pro_sanitize_display_rule_sett' );

/**
 * Filter to validate integration settings
 * 
 * @since 2.0
 */
function paoc_pro_sanitize_integration_sett( $input ) {

	// Integration Data
	$intgs_info	= get_option( 'paoc_intgs_info' );
	$intgs_info	= is_array( $intgs_info ) ? $intgs_info : (array)$intgs_info;

	/***** Mailchimp Integration Account Connection *****/
	if( ! empty( $_POST['paoc_intgs_mc_connect'] ) ) {

		// Integration Update Flag
		$integration_status = 1;

		// Mailchimp Settings
		$input['mc_api_key'] = isset( $input['mc_api_key'] ) ? paoc_pro_clean( $input['mc_api_key'] ) : '';

		if( $input['mc_api_key'] ) {

			// Initialize MailChimp
			$paoc_mc_api = new Paoc_Pro_MailChimp( $input['mc_api_key'] );

			// Get MailChimp Lists
			$mailchimp_lists	= $paoc_mc_api->get_lists(array('count' => 100));
			$mc_lists_arr		= array();

			if( is_array( $mailchimp_lists['lists'] ) ) {
				foreach ( $mailchimp_lists['lists'] as $key => $value ) {

					$list_id	= paoc_pro_clean( $value['id'] );
					$list_name	= paoc_pro_clean( $value['name'] );

					$mc_lists_arr[$list_id]['id']	= $list_id;
					$mc_lists_arr[$list_id]['name']	= $list_name;
				}
			}

			// Get MailChimp Account Information
			$mc_acc_info_data	= $paoc_mc_api->account_details();
			$mc_acc_name		= $mc_acc_info_data['account_name'];
			$mc_acc_email		= $mc_acc_info_data['email'];

			$intgs_info['mailchimp'] = array(
											'account_info' => array(
																'name'	=> $mc_acc_name,
																'email'	=> $mc_acc_email,
															),
											'lists'	=> $mc_lists_arr,
										);
		} else {

			$intgs_info['mailchimp'] = array();
		}

	} elseif ( ! empty( $_POST['paoc_intgs_mc_dconnect'] ) ) { // Disconnect Account

		// Integration Data
		$integration_status			= 1;
		$intgs_info['mailchimp']	= array();
	}

	// If any integration is updated
	if ( isset( $integration_status ) ) {
		update_option( 'paoc_intgs_info', $intgs_info, false );
		delete_option( 'paoc_mc_info' );
	}

	return $input;
}
add_filter( 'paoc_pro_sett_sanitize_integration', 'paoc_pro_sanitize_integration_sett' );