<?php
/**
 * Export Entries Functions
 *
 * @package Popup Anything on Click Pro
 * @since 2.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/**
 * Function to get file data if exist else create the new file
 * 
 * @since 2.0
 */
function paoc_pro_get_export_file( $file = false ) {

	$file_data = '';
	
	if ( $file && @file_exists( $file ) ) {

		$file_data = @file_get_contents( $file );

	} else {

		@file_put_contents( $file, '' );
		@chmod( $file, 0664 );
	}
	return $file_data;
}

/**
 * Append data to export file
 *
 * @since 2.0
 */
function paoc_pro_add_export_data( $file, $data = '' ) {

	$file_data = paoc_pro_get_export_file( $file );
	$file_data .= $data;
	@file_put_contents( $file, $file_data );
}

/**
 * Form Entries Export File
 *
 * @since 2.0
 */
function paoc_pro_export_file_path() {

	$upload_dir	= wp_upload_dir();
	$filename	= 'paoc-export-data.csv';
	$file		= trailingslashit( $upload_dir['basedir'] ) . $filename;

	return $file;
}

/**
 * Function to generate form entries CSV
 * 
 * @since 2.0
 */
function paoc_pro_do_export_action() {

	// Taking form data
	parse_str( $_POST['form_data'], $form_data );

	// Taking some variable
	$results = array(
						'status' 	=> 0,
						'message'	=> esc_html__('Sorry, Something happened wrong.', 'popup-anything-on-click')
					);
	$export_action	= isset( $form_data['export_action'] )	? paoc_pro_clean( $form_data['export_action'] )		: '';
	$nonce			= isset( $form_data['nonce'] )			? paoc_pro_clean( $form_data['nonce'] )				: '';
	$redirect_url	= isset( $form_data['redirect_url'] )	? paoc_pro_clean_url( $form_data['redirect_url'] )	: '';
	$page 			= isset( $_POST['page'] )				? $_POST['page']								: 1;
	$total_count 	= isset( $_POST['total_count'] )		? $_POST['total_count']							: 0;
	$data_process 	= isset( $_POST['data_process'] )		? $_POST['data_process']						: 0;

	if( empty( $_POST['form_data'] ) || empty( $redirect_url ) || ! wp_verify_nonce( $nonce, "paoc-export-nonce" ) ) {
		wp_send_json( $results );
	}

	// Gethering all data
	$form_data					= (array) $form_data;
	$form_data['limit']			= 30;
	$form_data['page']			= $page;
	$form_data['total_count']	= $total_count;
	$form_data['data_process']	= $data_process;

	if ( function_exists( 'paoc_'.$export_action ) ) {
		$results = call_user_func( 'paoc_'.$export_action, $form_data );
	} else {
		$results = paoc_pro_export_form_entries( $form_data );
	}

	wp_send_json( $results );
}

// Action to process Ajax for export data
add_action( 'wp_ajax_paoc_pro_do_export_action', 'paoc_pro_do_export_action' );

/**
 * Function to export form entries
 * 
 * @since 2.0
 */
function paoc_pro_export_form_entries( $args = array() ) {

	global $wpdb;

	// Taking some variables
	$col_data		= '';
	$row_data		= '';
	$file			= paoc_pro_export_file_path();
	$result			= array(
						'status'	=> 0,
						'message'	=> esc_html__( 'Sorry, No data found for export parameters.', 'popup-anything-on-click' )
					);

	$limit			= $args['limit'];
	$page			= ! empty( $args['page'] )			? $args['page']			: 1;
	$data_process	= isset( $args['data_process'] )	? $args['data_process']	: 0;
	$total_count	= isset( $args['total_count'] )		? $args['total_count']	: 0;

	$cols 		= paoc_pro_export_form_entry_columns( array(), $args );
	$csv_column	= count( $cols );

	// Getting data
	$entries_data = paoc_pro_export_data( array(), $args );

	// If data found
	if( $entries_data ) {

		// If process is newly started - Step 1
		if( $page < 2 ) {

			// Make sure we start with a fresh file on step 1
			if( file_exists( $file ) ) {
				@unlink( $file );
			}
			$i = 1;

			foreach( $cols as $col_id => $column ) {
				$col_data .= '"' . addslashes( $column ) . '"';
				$col_data .= ( $i == $csv_column ) ? '' : ',';
				$i++;
			}

			$col_data .= "\r\n";
			paoc_pro_add_export_data( $file, $col_data ); // Add columns to file

			// Taking total counts of data at first time
			$total_count = paoc_pro_export_data_count( 0, $args );
		}

		$count = 0;

		// Output each row
		foreach ( $entries_data as $row ) {
			$j = 1;
			$count++;

			foreach ( $cols as $col_id => $column ) {

				if ( isset( $row[ $col_id ] ) ) { // Make sure the column is valid

					$entry_data = paoc_pro_export_data_row( $row[ $col_id ], $col_id, $column, $args );

					$row_data .= '"' . addslashes( preg_replace( "/\"/","'", $entry_data ) ) . '"';
					$row_data .= $j == count( $cols ) ? '' : ',';
				}

				$j++;
			}
			$row_data .= "\r\n";
		} // End foreach

		// Record total process data
		$data_process = ( $data_process + count( $entries_data ) );
		paoc_pro_add_export_data( $file, $row_data ); // Add rows to file

		// Calculate percentage
		$percentage = 100;

		if( $total_count > 0 ) {
			$percentage = ( ( $limit * $page ) / $total_count ) * 100;
		}

		if( $percentage > 100 ) {
			$percentage = 100;
		}

		// If process is done
		if( $data_process >= $total_count ) {

			$result['url'] = add_query_arg( array( 'action' => 'paoc_process_export_file', '_wpnonce' => wp_create_nonce( 'paoc-download-export-file' ) ), $args['redirect_url'] );
		}

		$result['status'] 		= 1;
		$result['message']		= esc_html__('Entries exported successfully.', 'popup-anything-on-click');
		$result['page']			= ( $page + 1 );
		$result['total_count'] 	= $total_count;
		$result['percentage'] 	= $percentage;
		$result['data_process'] = $data_process;

	} else {
		return $result;
	}

	return $result;
}

/**
 * Process the download file generated by a batch export
 * 
 * @since 2.0
 */
function paoc_pro_download_export_file() {

	if( isset( $_GET['action'] ) && $_GET['action'] == 'paoc_process_export_file' && isset( $_GET['_wpnonce'] ) && wp_verify_nonce( $_GET['_wpnonce'], 'paoc-download-export-file' ) ) {

		// Set headers
		ignore_user_abort( true );
		set_time_limit( 0 );
		nocache_headers();

		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=popup-export-data-' . date( 'd-m-Y' ) . '.csv' );
		header( "Expires: 0" );

		$file_path 	= paoc_pro_export_file_path();
		$file 		= paoc_pro_get_export_file( $file_path );

		@unlink( $file_path );
		echo $file;
		exit();
	}
}

// Action to export download process
add_action( 'admin_init', 'paoc_pro_download_export_file', 5 );

/**
 * Function to return columns for export entries data
 * 
 * @since 2.0
 */
function paoc_pro_export_form_entry_columns( $cols, $args ) {

	// Taking some variables
	$prefix		= PAOC_PRO_META_PREFIX;
	$popup_id	= isset( $args['popup_id'] ) ? $args['popup_id'] : 0;

	// CSV Columns
	$cols = array(
		'id'			=> esc_html__( 'ID', 'popup-anything-on-click' ),
		'popup_id'		=> esc_html__( 'Popup ID', 'popup-anything-on-click' ),
		'popup_type'	=> esc_html__( 'Popup Type', 'popup-anything-on-click' ),
		'created_date'	=> esc_html__( 'Registered Date', 'popup-anything-on-click' ),
	);

	// If `Popup ID` is there
	if( $popup_id ) {

		// Taking content tab data
		$content		= get_post_meta( $popup_id, $prefix.'content', true );
		$form_fields	= isset( $content['form_fields'] ) ? $content['form_fields'] : array();

		foreach( $form_fields as $field_key => $field_data ) {

			// Skip data of `Captcha`
			if( $field_data['type'] == 'math_captcha' ) {
				continue;
			}

			$cols['meta_key_'.$field_key] = $field_data['label'];
		}

	} else { // Else 20 fields records for all popups

		$form_fields_clmn = apply_filters( 'paoc_pro_export_form_entries_columns', 20 );

		for ($i = 0; $i < $form_fields_clmn; $i++) {
			$cols["meta_key_{$i}"]	= esc_html__( 'Field', 'popup-anything-on-click' ) .' '. ($i+1);
		}
	}

	return $cols;
}

/**
 * Function to return total count of entries
 * 
 * @since 2.0
 */
function paoc_pro_export_data_count( $total_count, $args ) {

	$entries_data	= paoc_pro_get_export_entries( $args );
	$total_count	= $entries_data['count'];

	return $total_count;
}

/**
 * Function to return total data of entries
 * 
 * @since 2.0
 */
function paoc_pro_export_data( $data, $args ) {

	$entries_data	= paoc_pro_get_export_entries( $args );
	$data			= $entries_data['data'];

	return $data;
}

/**
 * Function to filter value for entry data
 * 
 * @since 2.0
 */
function paoc_pro_export_data_row( $data, $col_id, $column_name, $args ) {

	if( $col_id == 'phone' ) {

		$data = str_replace('+', ' +', $data);
		$data = str_replace('-', ' -', $data);

	} else if( $col_id == 'popup_type' ) {

		$popup_appear_data	= paoc_pro_when_appear_options();
		$data				= isset( $popup_appear_data[ $data ] ) ? $popup_appear_data[ $data ] : ucfirst( $data );
	}

	return $data;
}

/**
 * Get Export Entries
 * 
 * @since 2.0
 */
function paoc_pro_get_export_entries( $args = array() ) {

	global $wpdb;

	$entries			= array();
	$results			= array();
	$entry_ids			= array();
	$args['limit']		= ! empty( $args['limit'] )			? $args['limit']									: 1;
	$args['orderby']	= ! empty( $args['orderby'] )		? $args['orderby']									: 'created_date';
	$args['order']		= ! empty( $args['order'] )			? $args['order']									: 'ASC';
	$args['popup_id']	= ! empty( $args['popup_id'] )		? trim( $args['popup_id'] )							: '';
	$args['start_date']	= ! empty( $args['start_date'] )	? date( 'Y-m-d', strtotime($args['start_date']) )	: '';
	$args['end_date']	= ! empty( $args['end_date'] )		? date( 'Y-m-d', strtotime($args['end_date']) )		: '';

	if( ! empty( $args['page'] ) ) {
		$page = $args['page'];
	} else if ( ! empty( $_GET['paged'] ) ) {
		$page = $_GET['paged'];
	} else {
		$page = 1;
	}

	// Query Offset
	$page_offset = ( ( $page * $args['limit'] ) - $args['limit'] );

	// Get entry table data
	$sql = "SELECT SQL_CALC_FOUND_ROWS entry_tbl.* FROM `".PAOC_PRO_ENTRY_TBL."` AS entry_tbl
			INNER JOIN `".PAOC_PRO_ENTRY_META_TBL."` as entry_meta_tbl ON entry_tbl.`id` = entry_meta_tbl.`entry_id` WHERE 1=1";

	// Popup ID
	if( $args['popup_id'] ) {
		$sql .= " AND entry_tbl.`popup_id`='{$args['popup_id']}' ";
	}

	// Start Date
	if( $args['start_date'] ) {
		$sql .= " AND DATE(entry_tbl.`created_date`) >= '{$args['start_date']}' ";
	}

	// End Date
	if( $args['end_date'] ) {
		$sql .= " AND DATE(entry_tbl.`created_date`) <= '{$args['end_date']}' ";
	}

	// Group By
	$sql .= " GROUP BY entry_tbl.`id` ";

	// Order By
	if( $args['orderby'] ) {
		$sql .= " ORDER BY `{$args['orderby']}` {$args['order']} ";
	}

	// Limit
	if( $args['limit'] ) {
		$sql .= " LIMIT {$page_offset},{$args['limit']} ";
	}

	$form_entries = $wpdb->get_results( $sql, ARRAY_A );

	/***** Get Entry Meta *****/
	if( ! empty( $form_entries ) ) {

		// Total Entry Count
		$total_entries	= (int) $wpdb->get_var( 'SELECT FOUND_ROWS()' );

		// Loop of form entries
		foreach( $form_entries as $entry_key => $entry_data ) {
			
			// Store entry with indexed id
			$entries[ $entry_data['id'] ] = $entry_data;

			// Entry ID array
			$entry_ids[] = $entry_data['id'];
		}

		// Get Form Entries Meta
		$sql_entry_meta			= "SELECT * FROM ".PAOC_PRO_ENTRY_META_TBL." WHERE 1=1 AND entry_id IN(".implode( ',', $entry_ids ).")";
		$sql_entry_meta_result	= $wpdb->get_results( $sql_entry_meta, ARRAY_A );

		// Loop of form entries meta
		if( ! empty( $sql_entry_meta_result ) ) {
			foreach( $sql_entry_meta_result as $entry_key => $entry_data ) {

				if( isset( $entries[ $entry_data['entry_id'] ] ) ) {
					$entries[ $entry_data['entry_id'] ][ 'meta_key_'.$entry_data['meta_key'] ]		= $entry_data['meta_value'];
					$entries[ $entry_data['entry_id'] ][ 'meta_key_type_'.$entry_data['meta_key'] ] = $entry_data['meta_key_type'];
				}
			}
		}
	}

	$results['data']	= $entries;
	$results['count']	= ! empty( $total_entries ) ? $total_entries : 0;

	return $results;
}