<?php
/**
 * MailChimp API Library.
 *
 * @package Popup Anything on Click Pro
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class Paoc_Pro_MailChimp {

	/**
	 * MailChimp account API key.
	 *
 	 * @since 1.0
	 * @var    string $api_key MailChimp account API key.
	 */
	protected $api_key;

	/**
	 * MailChimp account data center.
	 *
 	 * @since 1.0
	 * @var    string $data_center MailChimp account data center.
	 */
	protected $data_center;

	/**
	 * Initialize API library.
	 *
 	 * @since 1.0
	 *
	 * @param string $api_key (default: '') MailChimp API key.
	 *
	 */
	public function __construct( $api_key = '' ) {

		$api_key = empty( $api_key ) ? paoc_pro_get_option( 'mc_api_key' ) : $api_key;

		// Assign API key to object.
		$this->api_key = $api_key;

		// Set data center.
		$this->set_data_center();
	}

	/**
	 * Get current account details.
	 *
 	 * @since 1.0
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function account_details() {

		return $this->process_request();
	}

	/**
	 * Get all interests for an interest category.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id     MailChimp list ID.
	 * @param string $category_id Interest category ID.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function get_interest_category_interests( $list_id, $category_id ) {

		return $this->process_request( 'lists/' . $list_id . '/interest-categories/' . $category_id . '/interests', array( 'count' => 9999 ), 'GET', 'interests' );
	}

	/**
	 * Get a specific MailChimp list.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id MailChimp list ID.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function get_list( $list_id ) {

		return $this->process_request( 'lists/' . $list_id );
	}

	/**
	 * Get all MailChimp lists.
	 *
 	 * @since 1.0
	 *
	 * @param array $params List request parameters.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function get_lists( $params ) {

		return $this->process_request( 'lists', $params );
	}

	/**
	 * Get all interest categories for a MailChimp list.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id MailChimp list ID.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function get_list_interest_categories( $list_id ) {

		return $this->process_request( 'lists/' . $list_id . '/interest-categories', array( 'count' => 9999 ), 'GET', 'categories' );
	}

	/**
	 * Get a specific MailChimp list member.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id       MailChimp list ID.
	 * @param string $email_address Email address.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function get_list_member( $list_id, $email_address ) {

		// Prepare subscriber hash.
		$subscriber_hash = md5( strtolower( $email_address ) );

		return $this->process_request( 'lists/' . $list_id . '/members/' . $subscriber_hash );
	}

	/**
	 * Get all merge fields for a MailChimp list.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id MailChimp list ID.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function get_list_merge_fields( $list_id ) {

		return $this->process_request( 'lists/' . $list_id . '/merge-fields', array( 'count' => 9999 ) );
	}

	/**
	 * Add or update a MailChimp list member.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id       MailChimp list ID.
	 * @param string $email_address Email address.
	 * @param array  $subscription  Subscription details.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function update_list_member( $list_id, $email_address, $subscription ) {

		// Prepare subscriber hash.
		$subscriber_hash = md5( strtolower( $email_address ) );

		return $this->process_request( 'lists/' . $list_id . '/members/' . $subscriber_hash, $subscription, 'PUT' );
	}

	/**
	 * Delete MailChimp list member.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id       MailChimp list ID.
	 * @param string $email_address Email address.
	 * @param array  $subscription  Subscription details.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function delete_list_member( $list_id, $email_address, $subscription ) {

		// Prepare subscriber hash.
		$subscriber_hash = md5( strtolower( $email_address ) );

		return $this->process_request( 'lists/' . $list_id . '/members/' . $subscriber_hash, $subscription, 'DELETE' );
	}

	/**
	 * Add a note to the MailChimp list member.
	 *
	 * @since  4.0.10
	 * @access public
	 *
	 * @param string $list_id       MailChimp list ID.
	 * @param string $email_address Email address.
	 * @param string $note          The note to be added to the member.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function add_member_note( $list_id, $email_address, $note ) {

		// Prepare subscriber hash.
		$subscriber_hash = md5( strtolower( $email_address ) );

		return $this->process_request( 'lists/' . $list_id . '/members/' . $subscriber_hash . '/notes', array( 'note' => $note ), 'POST' );
	}

	/**
	 * Add tag to a MailChimp list member.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id       MailChimp list ID.
	 * @param string $email_address Email address.
	 * @param array  $tags  		tag details.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function add_member_tag( $list_id, $email_address, $tags ) {

		// Prepare subscriber hash.
		$tag_data			= array();
		$subscriber_hash	= md5( strtolower( $email_address ) );

		if ( ! is_array( $tags ) ) {
			$tag_arr = explode( ',', $tags );

			if ( ! empty( $tag_arr ) ) {
				foreach( $tag_arr as $tag_key => $tag_val ) {
					$tag_data['tags'][] = array(
											'name' 		=> trim( $tag_val ),
											'status'	=> 'active',
										);
				}
			}
		} else {
			$tag_data = $tags;
		}

		$tag_data = apply_filters( 'paoc_pro_mc_update_tags_data', $tag_data, $list_id, $email_address );

		return $this->process_request( 'lists/' . $list_id . '/members/' . $subscriber_hash . '/tags', $tag_data, 'POST' );
	}

	/**
	 * Process MailChimp API request.
	 *
 	 * @since 1.0
	 *
	 * @param string $path       Request path.
	 * @param array  $data       Request data.
	 * @param string $method     Request method. Defaults to GET.
	 * @param string $return_key Array key from response to return. Defaults to null (return full response).
	 *
	 * @throws Exception if API request returns an error, exception is thrown.
	 *
	 * @return array
	 */
	private function process_request( $path = '', $data = array(), $method = 'GET', $return_key = null ) {

		// If API key is not set, throw exception.
		if ( empty( $this->api_key ) ) {
			return false;
		}

		// Build base request URL.
		$request_url = 'https://' . $this->data_center . '.api.mailchimp.com/3.0/' . $path;

		// Add request URL parameters if needed.
		if ( 'GET' === $method && ! empty( $data ) ) {
			$request_url = add_query_arg( $data, $request_url );
		}

		// Build base request arguments.
		$args = array(
			'method'   => $method,
			'headers'  => array(
				'Accept'        => 'application/json',
				'Authorization' => 'Basic ' . base64_encode( ':' . $this->api_key ),
				'Content-Type'  => 'application/json',
			),
			'sslverify' => false,
			'timeout'   => 30,
		);

		// Add data to arguments if needed.
		if ( 'GET' !== $method ) {
			$args['body'] = json_encode( $data );
		}

		// Get request response.
		$response = wp_remote_request( $request_url, $args );

		// Get the response code.
		$response_code = wp_remote_retrieve_response_code( $response );

		if ( $response_code == 200 ) {

			// Decode response body.
			$response['body'] = json_decode( $response['body'], true );

			// Remove links from response.
			unset( $response['body']['_links'] );

			// If a return key is defined and array item exists, return it.
			if ( ! empty( $return_key ) && isset( $response['body'][ $return_key ] ) ) {
				return $response['body'][ $return_key ];
			}

			return $response['body'];
		}
	}

	/**
	 * Set data center based on API key.
	 *
 	 * @since 1.0
	 */
	private function set_data_center() {

		// If API key is empty, return.
		if ( empty( $this->api_key ) ) {
			return;
		}

		// Explode API key.
		$exploded_key = explode( '-', $this->api_key );

		// Set data center from API key.
		$this->data_center = isset( $exploded_key[1] ) ? $exploded_key[1] : 'us1';
	}

	/**
	 * Add or update a MailChimp list member.
	 *
 	 * @since 1.0
	 *
	 * @param string $list_id       MailChimp list ID.
	 * @param string $email_address Email address.
	 * @param array  $subscription  Subscription details.
	 *
	 * @uses Paoc_Pro_MailChimp::process_request()
	 *
	 * @return array
	 */
	public function update_data_in_list( $lists, $args ) {

		// Taking some variable for `MailChimp`
		$mc_lists = is_array( $lists ) ? $lists : (array)$lists;
		
		// Taking some data
		$email			= ! empty( $args['email'] )			? $args['email']		: '';
		$merge_fields	= ! empty( $args['merge_fields'] )	? $args['merge_fields'] : array();

		if( empty( $this->api_key ) || empty( $mc_lists ) || empty( $email ) ) {
			return false;
		}

		// MailChimp Lists ID Loop
		foreach ( $mc_lists as $mc_lists_id ) {

			// Update list filter for particular
			$update_list_data = apply_filters( 'paoc_pro_mc_update_list_data', array(
																					'status'		=> empty( $args['status'] ) ? 'subscribed' : 'pending',
																					'email_address'	=> $email,
																					'merge_fields'	=> $merge_fields,
																				), $mc_lists_id, $args );

			$this->update_list_member( $mc_lists_id, $email, $update_list_data );
		}
	}
}