<?php
/**
 * Public Class
 *
 * Handles the Public side functionality of plugin
 *
 * @package Popup Anything on Click Pro
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

class PAOC_PRO_Public {

	function __construct() {

		// Render Popup
		add_action( 'wp_footer', array($this, 'paoc_pro_render_popup') );

		// Add action to detect adblocker
		add_action( 'wp_footer', array($this, 'paoc_pro_detect_adblocker'), 99 );

		// Add action to store popup views
		add_action( 'wp_ajax_paoc_pro_store_popup_click_view', array($this, 'paoc_pro_store_popup_click_view') );
		add_action( 'wp_ajax_nopriv_paoc_pro_store_popup_click_view', array($this,'paoc_pro_store_popup_click_view') );

		// Add action to process popup form submission
		add_action( 'wp_ajax_paoc_pro_popup_form_submit', array($this, 'paoc_pro_popup_form_submit') );
		add_action( 'wp_ajax_nopriv_paoc_pro_popup_form_submit', array($this,'paoc_pro_popup_form_submit') );

		// Add action to get geolocation data
		add_action( 'wp_ajax_paoc_pro_process_popup_data', array($this, 'paoc_pro_process_popup_data') );
		add_action( 'wp_ajax_nopriv_paoc_pro_process_popup_data', array($this,'paoc_pro_process_popup_data') );

		// Render Popup Preview
		add_action( 'wp', array($this, 'paoc_pro_render_popup_preview') );
	}

	/**
	 * Function to render popup
	 * 
	 * @since 2.0
	 */
	function paoc_pro_render_popup() {

		global $post, $paoc_preview, $paoc_popup_data;

		// If preview is there then simple bypass it
		if( $paoc_preview ) {
			$this->paoc_pro_create_popup();
			return false;
		}

		// Taking some data
		$skip_gen_popup	= array();
		$prefix			= PAOC_PRO_META_PREFIX;
		$post_id		= isset( $post->ID ) ? $post->ID : '';
		$post_type		= isset( $post->post_type ) ? $post->post_type : '';
		$enable			= paoc_pro_get_option( 'enable' );
		$enable			= apply_filters( 'paoc_pro_render_popup', $enable, $post );

		// If not globally enable
		if ( ! $enable ) {
			return false;
		}

		/***** Start - Display Welcome Popup *****/
		$enabled_post_types	= paoc_pro_get_option( 'post_types', array() );
		$welcome_popup		= get_post_meta( $post_id, $prefix.'welcome_popup', true );

		// Check Post Type Meta else Global
		if( $welcome_popup && in_array( $post_type, $enabled_post_types ) ) {

			$enable_welcome_popup = true;

		} else {

			$welcome_popup			= paoc_pro_get_option( 'welcome_popup' );
			$welcome_display_in 	= paoc_pro_get_option( 'welcome_display_in' );
			$enable_welcome_popup	= paoc_pro_check_active( $welcome_display_in );
		}

		// Render Campaign Popup
		if( $welcome_popup > 0 && $enable_welcome_popup ) {
	
			$welcome_popup_data = (array)$this->paoc_pro_campaign_display_popup( $welcome_popup );

			foreach( $welcome_popup_data as $welcome_popup_key => $welcome_popup ) {
				$skip_gen_popup[] = $welcome_popup;

				$this->paoc_pro_create_popup( $welcome_popup );
			}
		}
		/***** End - Display Welcome Popup *****/


		/***** Start - Display Exit Intent *****/
		$exit_popup	= get_post_meta( $post_id, $prefix.'exit_popup', true );

		// Check Post Type Meta else Global
		if( $exit_popup && in_array( $post_type, $enabled_post_types ) ) {

			$enable_exit_popup = true;

		} else {

			$exit_popup			= paoc_pro_get_option( 'exit_popup' );
			$exit_display_in 	= paoc_pro_get_option( 'exit_display_in' );
			$enable_exit_popup	= paoc_pro_check_active( $exit_display_in );
		}

		// Render Campaign Popup
		if( $exit_popup > 0 && $enable_exit_popup ) {
			$exit_popup_data = (array)$this->paoc_pro_campaign_display_popup( $exit_popup );

			foreach( $exit_popup_data as $exit_popup_key => $exit_popup ) {
				$skip_gen_popup[] = $exit_popup;
				$this->paoc_pro_create_popup( $exit_popup );
			}
		}
		/***** End - Display Exit Intent Popup *****/


		/***** Start - Display General Popup *****/
		$general_popup	= get_post_meta( $post_id, $prefix.'general_popup', true );

		// Check Post Type Meta else Global
		if( $general_popup && in_array( $post_type, $enabled_post_types ) ) {

			$enable_general_popup = true;

		} else {

			$general_popup			= paoc_pro_get_option( 'general_popup' );
			$general_display_in 	= paoc_pro_get_option( 'general_display_in' );
			$enable_general_popup	= paoc_pro_check_active( $general_display_in );
		}

		if( ( $general_popup > 0 || ! empty( $general_popup ) ) && $enable_general_popup ) {

			foreach ( $general_popup as $general_key => $general_val ) {

				/* If `Welcome Popup` & `Exit Popup` is already there to skip from the `General Popup` */
				if( in_array( $general_val, $skip_gen_popup ) ) {
					continue;
				}

				// Render Campaign Popup
				$general_popup_data = (array)$this->paoc_pro_campaign_display_popup( $general_val );

				foreach( $general_popup_data as $gen_popup_key => $gen_popup_val ) {

					$skip_gen_popup[] = $gen_popup_val;
					$this->paoc_pro_create_popup( $gen_popup_val );
				}
			}
		}
		/***** End - Display General Popup *****/

		// Check for Popup( Simple Link, Button & Image )
		if( ! empty( $paoc_popup_data ) ) {

			foreach ( $paoc_popup_data as $popup_id => $popup_data ) {

				$paoc_popup_data[ $popup_id ]['render'] = 1;

				// Skip same popup if already set in `Welcome Popup`, `Exit Popup` OR `General Popup`
				if( in_array( $popup_id, $skip_gen_popup ) ) {
					continue;
				}

				if( $popup_data['popup_type'] == 'popup' ) {

					// Render Campaign Popup
					$shrt_popup_data = (array)$this->paoc_pro_campaign_display_popup( $popup_id );

					foreach( $shrt_popup_data as $shrt_popup_key => $shrt_popup_val ) {
						$this->paoc_pro_create_popup( $shrt_popup_val );
					}
				}
			}

			$paoc_popup_data = ''; // Flushing the global popup data variable
		}
	}

	/**
	 * Function to create nested popup HTML
	 * 
	 * @since 2.0.5
	 */
	function paoc_pro_render_nested_popup( $paoc_global_diff = array() ) {

		global $paoc_popup_data;

		// Check for Popup( Simple Link, Button & Image )
		if( ! empty( $paoc_global_diff ) ) {

			foreach ( $paoc_global_diff as $popup_id => $popup_data ) {

				// Continue already rendered popup
				if ( ! empty( $popup_data['render'] ) ) {
					continue;
				}

				$paoc_popup_data[ $popup_id ]['render'] = 1;

				$this->paoc_pro_create_popup( $popup_id );
			}
		}
	}

	/**
	 * Function to create popup HTML
	 * 
	 * @since 2.0
	 */
	function paoc_pro_create_popup( $popup_id = 0 ) {

		global $current_user, $paoc_preview, $paoc_campaign_popup_data, $paoc_design_sett, $paoc_behaviour_sett, $paoc_advance_sett, $paoc_custom_css, $paoc_adblocker, $paoc_popup_post, $paoc_popup_data;

		/**
		 * Store popup global data in temp variable before `do_shortcode`
		 * This is used to avoid rendering same popup multiple times in DOM
		 */
		$paoc_popup_data	= $paoc_popup_data ? $paoc_popup_data : array();
		$paoc_global_temp_1 = $paoc_popup_data;

		// If Popup Preview is there
		if( $paoc_preview == 1 ) {
			$popup_id = isset( $_POST['paoc_preview_popup_id'] ) ? $_POST['paoc_preview_popup_id'] : $popup_id;
		}

		// Return If no popup ID is there
		if( empty( $popup_id ) ) {
			return false;
		}

		// Query args
		$args = array(
				'post_type'			=> PAOC_PRO_POST_TYPE,
				'post__in'			=> array( $popup_id ),
				'post_status'		=> ( ! $paoc_preview ) ? array( 'publish' ) : array( 'any', 'auto-draft', 'inherit', 'trash' ),
				'posts_per_page'	=> 1,
				'no_found_rows'		=> true,
			);

		// WP Query for Popup
		$popup_query = get_posts( $args );

		// If no query post found
		if ( ! $popup_query ) {
			return false;
		}

		// Taking some data
		$prefix				= PAOC_PRO_META_PREFIX;
		$current_time		= current_time( 'timestamp' );
		$advance 			= paoc_pro_get_meta( $popup_id, $prefix.'advance' );
		$show_for			= isset( $advance['show_for'] )		? $advance['show_for']	: '';
		$allowed_roles		= ! empty( $advance['role'] )		? $advance['role']		: array();
		$current_user_role	= ! empty( $current_user->roles ) 	? $current_user->roles	: array();

		if( ( $show_for == 'guest' && is_user_logged_in() )		// Check guest user
			|| ( $show_for == 'member' && ( ! is_user_logged_in() || ($allowed_roles && ! array_intersect( $current_user_role, $allowed_roles )) ) ) )	// Check current user with show for member
		{
			return false;
		}

		// Taking some data
		$animateFrom	= 'top';
		$animateTo		= 'top';
		$add_legacy_js	= paoc_pro_get_option( 'add_js' );
		$geo_enable		= paoc_pro_get_option( 'geo_enable' );
		$behaviour 		= paoc_pro_get_meta( $popup_id, $prefix.'behaviour' );
		$popup_goal 	= paoc_pro_get_meta( $popup_id, $prefix.'popup_goal' );
		$display_type 	= paoc_pro_get_meta( $popup_id, $prefix.'display_type' );
		$popup_appear 	= paoc_pro_get_meta( $popup_id, $prefix.'popup_appear' );
		$design 		= paoc_pro_get_meta( $popup_id, $prefix.'design' );
		$social 		= paoc_pro_get_meta( $popup_id, $prefix.'social' );
		$content 		= paoc_pro_get_meta( $popup_id, $prefix.'content' );
		$analytics 		= paoc_pro_get_meta( $popup_id, $prefix.'analytics' );
		$custom_css 	= paoc_pro_get_meta( $popup_id, $prefix.'custom_css' );

		// Campaign popup data
		$campaign_id		= paoc_pro_get_meta( $popup_id, $prefix.'campaign_id' );
		$campaign_id		= ! empty( $campaign_id )												? $campaign_id													: 0;
		$camp_popup_ids		= ! empty( $paoc_campaign_popup_data[ $campaign_id ]['display_popup'] )	? $paoc_campaign_popup_data[ $campaign_id ]['display_popup']	: 0;
		$campaign_enable	= ! empty( $camp_popup_ids ) ? 1 : 0;

		// Assign value to global var
		$paoc_popup_post		= $popup_query[0];
		$paoc_design_sett		= $design;
		$paoc_behaviour_sett	= $behaviour;
		$paoc_advance_sett		= $advance;
		$paoc_custom_css		= $custom_css;
		$paoc_adblocker			= ( ! empty( $advance['adblocker'] ) && empty( $paoc_adblocker ) ) ? $advance['adblocker'] : $paoc_adblocker;

		// Taking design data
		$template			= isset( $design['template'] )				? $design['template']				: 'design-1';
		$width				= isset( $design['width'] )					? $design['width']					: '';
		$height				= isset( $design['height'] )				? $design['height']					: '';
		$effect				= ! empty( $design['effect'] )				? $design['effect']					: 'fadein';
		$loader_color		= ! empty( $design['loader_color'] )		? $design['loader_color']			: '#000000';
		$popup_img			= isset( $design['popup_img'] )				? $design['popup_img']				: '';
		$speed_in			= ! empty( $design['speed_in'] )			? ( $design['speed_in'] * 1000 )	: 0.5;
		$speed_out			= ! empty( $design['speed_out'] )			? ( $design['speed_out'] * 1000 )	: 0.25;
		$fullscreen_popup	= ! empty( $design['fullscreen_popup'] )	? true								: false;

		// Taking Behaviour data
		$open_delay					= ! empty( $behaviour['open_delay'] )		? ( $behaviour['open_delay'] * 1000 )	: 0;
		$loader_speed				= ! empty( $behaviour['loader_speed'] )		? ( $behaviour['loader_speed'] * 1000 )	: 1000;
		$disappear					= ! empty( $behaviour['disappear'] )		? ( $behaviour['disappear'] )			: 0;
		$open_inactivity			= ! empty( $behaviour['open_inactivity'] )	? $behaviour['open_inactivity']			: 0;
		$open_scroll				= ! empty( $behaviour['open_scroll'] )		? $behaviour['open_scroll']				: 0;		
		$enable_loader				= ! empty( $behaviour['enable_loader'] )	? true	: false;
		$hide_close 				= ! empty( $behaviour['hide_close'] )		? true	: false;
		$clsonesc					= ! empty( $behaviour['clsonesc'] )			? true	: false;
		$close_overlay				= ! empty( $behaviour['close_overlay'] )	? true	: false;
		$hide_overlay				= ! empty( $behaviour['hide_overlay'] )		? false	: true;
		$atts['close_icon_tooltip']	= isset( $behaviour['close_icon_tooltip'] ) ? $behaviour['close_icon_tooltip']	: '';

		$atts['popup_position']		= '';
		$atts['type']				= 'popup';
		$atts['popup_appear']		= $popup_appear;
		$atts['popup_goal']			= $popup_goal;
		$atts['template']			= $template;
		$atts['popup_id']			= $popup_id;
		$atts['hide_close']			= $hide_close;
		$atts['display_type']		= $display_type;
		$atts['popup_appear_cls']	= str_replace('_', '-', $popup_appear);
		$atts['unique']				= paoc_pro_get_unique();
		$atts['style']				= paoc_pro_generate_popup_style( $popup_id );
		$atts['popup_classes']		= "paoc-popup-{$atts['popup_id']} paoc-popup-{$atts['popup_appear_cls']} paoc-popup-{$atts['popup_goal']} paoc-popup-{$atts['popup_goal']}-{$atts['template']} paoc-{$atts['template']}";		

		// Taking content data
		$atts['main_heading']		= isset( $content['main_heading'] )		? $content['main_heading']		: '';
		$atts['sub_heading']		= isset( $content['sub_heading'] )		? $content['sub_heading']		: '';
		$atts['cust_close_txt']		= isset( $content['cust_close_txt'] )	? $content['cust_close_txt']	: '';
		$atts['security_note']		= isset( $content['security_note'] )	? $content['security_note']		: '';

		/* Social tab data */
		$atts['social_type'] = isset( $social['social_type'] )		? $social['social_type'] : '';
		$atts['social_data'] = isset( $social['social_traffic'] )	? paoc_pro_get_social_data( $social['social_traffic'] ) : '';

		// Advance Tab Data
		$display_on			= isset( $advance['display_on'] )			? $advance['display_on']	: '';
		$referrer_mode		= isset( $advance['referrer_mode'] )		? $advance['referrer_mode'] : '';
		$utm_mode			= isset( $advance['utm_mode'] )				? $advance['utm_mode']		: '';
		$referrer_url		= isset( $advance['referrer_url'] )			? $advance['referrer_url'] 	: '';
		$utm_params			= isset( $advance['utm_params'] )			? $advance['utm_params'] 	: '';
		$cookie_params		= isset( $advance['cookie_params'] )		? $advance['cookie_params'] : '';
		$cookie_expire		= isset( $advance['cookie_expire'] )		? $advance['cookie_expire']	: '';
		$cookie_unit		= ! empty( $advance['cookie_unit'] )		? $advance['cookie_unit'] 	: 'day';
		$geo_countries		= ! empty( $advance['geo_countries'] )		? $advance['geo_countries'] : array();
		$adblocker			= ! empty( $advance['adblocker'] )			? $advance['adblocker']		: 0;
		$referrer_enable	= ! empty( $advance['referrer_enable'] )	? 1	: 0;
		$utm_enable			= ! empty( $advance['utm_enable'] )			? 1	: 0;
		$cookie_enable		= ! empty( $advance['cookie_enable'] )		? 1	: 0;
		$store_no_views		= ! empty( $advance['store_no_views'] )		? 1	: 0;
		$store_no_views		= ( $campaign_enable && $campaign_id )		? 0	: $store_no_views;
		$store_no_data		= ! empty( $advance['store_no_data'] )		? 1	: 0;

		$atts['show_credit']	= ! empty( $advance['show_credit'] )	? 1	: 0;
		$atts['credit_link']	= add_query_arg( array('utm_source' => site_url(), 'utm_medium' => 'popup', 'utm_campaign' => 'credit-link'), PAOC_PRO_LINK );

		// Taking schedule data
		$recurring_data		= paoc_pro_recurring_dates( $popup_id );
		$popup_start_date	= ! empty( $advance['start_time'] )		? $advance['start_time']		: '';
		$popup_end_date		= ! empty( $advance['end_time'] )		? $advance['end_time']			: '';
		$schedule_mode		= ! empty( $advance['schedule_mode'] )	? $advance['schedule_mode']		: 'default';
		$recurring_type		= ! empty( $advance['recurring_type'] )	? $advance['recurring_type']	: 'daily';

		// Add schedule class
		if( $popup_start_date || $popup_end_date || $schedule_mode == 'recurring' ) {
			$atts['popup_classes'] .= ' paoc-popup-has-schedule';
		}

		// Google Analytics Tab Data
		$ga_action		= isset( $analytics['action'] )		? $analytics['action']		: '';
		$ga_category	= isset( $analytics['category'] )	? $analytics['category']	: '';
		$ga_label		= isset( $analytics['label'] )		? $analytics['label']		: '';
		$ga_enable		= ! empty( $analytics['enable'] )	? 1	: 0;

		// If disappear value is negative
		if( $disappear < 0 ) {
			$disappear_mode = 'force';
			$disappear		= abs( $disappear );
		} else { // Else disappear value is possitive
			$disappear_mode = 'normal';
		}

		// Taking some variable
		$width = ! empty( $fullscreen_popup ) ? '100%'	: $width;

		/* Popup Goal : Collect Lead */
		if( $popup_goal == 'email-lists' ) {

			// Math Captcha
			$captcha_token	= wp_salt( 'auth' );
			$captcha_digit1	= mt_rand(1, 10);
			$captcha_digit2	= mt_rand(1, 89);

			if( mt_rand(0, 1) === 1 ) {
				$atts['paoc_captcha']	= "$captcha_digit1 + $captcha_digit2";
				$captcha_sum 			= $captcha_digit1 + $captcha_digit2;
			} else {
				$atts['paoc_captcha']	= "$captcha_digit2 - $captcha_digit1";
				$captcha_sum 			= $captcha_digit2 - $captcha_digit1;
			}

			$atts['captcha_val']	= sha1( $captcha_token . $captcha_sum ); // Hashing algorithm
			$atts['form_fields']	= isset( $content['form_fields'] )		? $content['form_fields']		: '';
			$atts['submit_btn_txt']	= ! empty( $content['submit_btn_txt'] )	? $content['submit_btn_txt']	: __('Submit', 'popup-anything-on-click');
		}

		/* Popup Goal : Target URL */
		if( $popup_goal == 'target-url' ) {
			$atts['btn1_text']		= isset( $content['target_url']['btn1_text'] )		? $content['target_url']['btn1_text']		: '';
			$atts['btn1_link']		= isset( $content['target_url']['btn1_link'] )		? $content['target_url']['btn1_link']		: '';
			$atts['btn1_target']	= isset( $content['target_url']['btn1_target'] )	? $content['target_url']['btn1_target']		: '';

			$atts['btn2_text']		= isset( $content['target_url']['btn2_text'] )		? $content['target_url']['btn2_text']		: '';
			$atts['btn2_link']		= isset( $content['target_url']['btn2_link'] )		? $content['target_url']['btn2_link']		: '';
			$atts['btn2_target']	= isset( $content['target_url']['btn2_target'] )	? $content['target_url']['btn2_target']		: '';

			$atts['btn1_click']		= 'onclick="window.open(\''.$atts['btn1_link'].'\', \''.$atts['btn1_target'].'\');"';
			
			if( $atts['btn2_link'] ) {
				$atts['btn2_click']	= 'onclick="window.open(\''.$atts['btn2_link'].'\', \''.$atts['btn2_target'].'\');"';
			} else {
				$atts['btn2_click']	= 'onclick="Custombox.modal.close();"';
			}
		}

		/* Popup Goal : Phone Call */
		if( $popup_goal == 'phone-calls' ) {
			$atts['btn_txt']	= isset( $content['phone_calls']['btn_txt'] )	? $content['phone_calls']['btn_txt']	: '';
			$atts['phone_num']	= isset( $content['phone_calls']['phone_num'] )	? $content['phone_calls']['phone_num']	: '';
		}

		/* Display Type : Bar */
		if( $display_type == 'bar' ) {

			$positionX				= 'center';
			$atts['template']		= ! empty( $design['bar_template'] )	? $design['bar_template'] : $atts['template'];
			$atts['popup_position']	= ! empty( $design['bar_position'] )	? $design['bar_position'] : 'top';
			$atts['popup_classes']	.= " paoc-bar-{$atts['template']}";

			if( $atts['popup_position'] == 'top' || $atts['popup_position'] == 'top-push' ) {
				$positionY	= 'top';
			} else if( $atts['popup_position'] == 'bottom' || $atts['popup_position'] == 'bottom-push' ) {
				$positionY	= 'bottom';
			}
		}

		/* Display Type : Modal & Push Notification */
		if( $display_type == 'modal' || $display_type == 'push-notification' || $display_type == 'slide-in' ) {

			$atts['popup_classes'] .= " paoc-modal-{$atts['template']}";

			$atts['popup_position'] = ! empty( $design['mn_position'] ) ? $design['mn_position'] : 'center-center';
			$popup_position			= explode('-', $atts['popup_position']);
			$positionX				= $popup_position[0];
			$positionY				= $popup_position[1];
		}

		$atts['popup_classes'] .= " paoc-position-{$atts['popup_position']}";

		/* Display Type : Modal */
		if( $display_type == 'modal' || $display_type == 'bar' ) {
			$atts['popup_classes']	.= " paoc-popup-js";
		}

		/* Display Type : Push Notification */
		if( $display_type == 'push-notification' ) {
			$positionX				= 'center';
			$positionY				= 'center';
			$atts['popup_classes']	.= " paoc-nfs-popup paoc-notif-popup";
		}

		/* Display Type : Slide In */
		if( $display_type == 'slide-in' ) {

			// Taking some variable
			$effect					= 'slide';
			$atts['slide_in_text']	= isset( $content['slide_in_text'] )		? $content['slide_in_text']		: '';
			$atts['popup_classes']	.= " paoc-nfs-popup paoc-slide-popup";

			// Popup Position is `Top`
			if( $atts['popup_position'] == 'center-top' || $atts['popup_position'] == 'center-center' ) {
				$animateFrom	= 'top';
				$animateTo		= 'top';
			}

			// Popup Position is `Bottom`
			if( $atts['popup_position'] == 'center-bottom' ) {
				$animateFrom	= 'bottom';
				$animateTo		= 'bottom';
			}

			// Popup Position is `Left`
			if( $atts['popup_position'] == 'left-top' || $atts['popup_position'] == 'left-center' || $atts['popup_position'] == 'left-bottom' ) {
				$animateFrom	= 'left';
				$animateTo		= 'left';
			}

			// Popup Position is `Right`
			if( $atts['popup_position'] == 'right-top' || $atts['popup_position'] == 'right-center' || $atts['popup_position'] == 'right-bottom' ) {
				$animateFrom	= 'right';
				$animateTo		= 'right';
			}
		}

		// If `Referer Popup` is there
		if( $referrer_enable && ! empty( $referrer_url ) ) {

			$referrer_url_arr = explode( "\n", $referrer_url );

			foreach ($referrer_url_arr as $referrer_url_key => $referrer_url_val) {
				
				$referrer_url_data	= explode('/', $referrer_url_val);
				$referrer_url_data	= array_map( 'urlencode', $referrer_url_data );
				$referrer_url_val	= implode('/', $referrer_url_data);

				$data_domain[ $referrer_url_key ] = strtolower( $referrer_url_val );
			}

			$atts['popup_classes'] .= " paoc-popup-has-referrer";
		}

		// If `UTM Popup` is there
		if( $utm_enable && ! empty( $utm_params ) ) {

			$utm_param_arr = explode( "\n", $utm_params );

			if( ! empty( $utm_param_arr ) ) {
				foreach ($utm_param_arr as $utm_param_key => $utm_param_val) {

					$utm_par = explode( '|', $utm_param_val );
					$utm_key = trim( $utm_par[0] );
					$utm_val = esc_attr( trim( $utm_par[1] ) );

					$data_utm[$utm_key][] = $utm_val;
				}
			}

			$atts['popup_classes'] .= " paoc-popup-has-utm";
		}

		/* If `Cookie Popup` is there */
		if( $cookie_enable && ! empty( $cookie_params ) ) {

			$cookie_param_arr = explode( "\n", $cookie_params );

			if( ! empty( $cookie_param_arr ) ) {
				foreach ($cookie_param_arr as $cookie_param_key => $cookie_param_val) {

					$cookie_par	= explode( '|', $cookie_param_val );
					$cookie_key	= trim( $cookie_par[0] );
					$cookie_val	= esc_attr( trim( $cookie_par[1] ) );

					$data_cookie[$cookie_key][] = $cookie_val;
				}
			}

			$atts['popup_classes'] .= " paoc-popup-has-cookie";
		}

		// Creating Popup Configuration
		$data_conf = array(
						'id'				=> $popup_id,
						'popup_type'		=> $popup_appear,
						'display_type'		=> $display_type,
						'disappear'			=> $disappear,
						'disappear_mode'	=> $disappear_mode,
						'open_delay'		=> $open_delay,
						'open_inactivity'	=> $open_inactivity,
						'open_scroll'		=> $open_scroll,
						'popup_position'	=> $atts['popup_position'],
						'store_no_data'		=> $store_no_data,
						'store_no_views'	=> $store_no_views,
						'cookie_expire'		=> $cookie_expire,
						'cookie_unit'		=> $cookie_unit,
						'display_on'		=> $display_on,
						'referrer_enable'	=> $referrer_enable,
						'referrer_mode'		=> $referrer_mode,
						'utm_enable'		=> $utm_enable,
						'utm_mode'			=> $utm_mode,
						'campaign_enable'	=> $campaign_enable,
						'campaign_id'		=> $campaign_id,
						'campaign_popup'	=> $camp_popup_ids,
						'adblocker'			=> $adblocker,
						'geo_countries'		=> $geo_countries,
						'popup_start_date'	=> $popup_start_date,
						'popup_end_date'	=> $popup_end_date,
						'schedule_mode'		=> $schedule_mode,
						'recurring_data'	=> $recurring_data,
						'utm_params'		=> isset( $data_utm )		? $data_utm		: array(),
						'cookie_params'		=> isset( $data_cookie )	? $data_cookie	: array(),
						'referrer_url'		=> isset( $data_domain )	? $data_domain	: array(),
					);

		// Creating Popup Configuration
		$popup_conf = array();
		$popup_conf['content'] = array(
									'target'		=> "#paoc-popup-{$popup_id}-{$atts['unique']}",
									'effect' 		=> $effect,
									'positionX'		=> $positionX,
									'positionY'		=> $positionY,
									'fullscreen'	=> $fullscreen_popup,
									'speedIn'		=> $speed_in,
									'speedOut'		=> $speed_out,
									'close'			=> $clsonesc,
									'animateFrom'	=> $animateFrom,
									'animateTo'		=> $animateTo,
								);
		$popup_conf['loader'] = array(
									'active' 	=> $enable_loader,
									'color'		=> $loader_color,
									'speed'		=> $loader_speed,
								);
		$popup_conf['overlay'] = array(
									'active' 	=> $hide_overlay,
									'color'		=> 'rgba(0, 0, 0, 0.5)',
									'close' 	=> $close_overlay,
									'opacity'	=> 1,
								);

		$data_conf	= htmlspecialchars( json_encode( $data_conf ) );
		$popup_conf	= htmlspecialchars( json_encode( $popup_conf ) );

		// Set Data Attribute
		$atts['popup_attr'] = "data-popup-conf='{$popup_conf}' data-conf='{$data_conf}' data-id='paoc-popup-{$atts['popup_id']}'";

		// Set icon classes for Push Notification & Slide In
		$atts['panel_classes'] = "paoc-popup-panel-{$popup_id} paoc-{$atts['popup_position']}";

		/* Popup image is there */
		$atts['popup_classes'] .= ( $popup_img ) ? " paoc-has-image" : " paoc-no-image";

		/* Campaign is there */
		if( $campaign_enable ) {
			$atts['popup_classes'] .= " paoc-campaign-popup paoc-campaign-{$campaign_id} paoc-campaign-popup-{$popup_id}";
			$atts['panel_classes'] .= " paoc-campaign-panel-{$campaign_id}";
		}

		/* Google Analytics is there */
		if( $ga_enable ) {
			$atts['popup_attr'] .= " data-label='".esc_attr( $ga_label )."' data-event='".esc_attr( $ga_category )."' data-event-action='".esc_attr( $ga_action )."'";
		}

		// If full screen popup is there
		if( $fullscreen_popup ) {
			$atts['popup_classes'] .= " paoc-popup-fullscreen";
		}

		/* If `Geo Countries` is there */
		if( ! empty( $geo_enable ) && $geo_countries ) {
			$atts['popup_classes'] .= " paoc-popup-has-geolocation";
		}

		// Enquque Scripts
		wp_enqueue_script('jquery');

		if( empty( $add_legacy_js ) ) {
			wp_enqueue_script('wpos-custombox-legacy-js');
		}

		if ( $adblocker ) {
			wp_enqueue_script('paoc-ads-prebid');
		}

		wp_enqueue_script('wpos-custombox-popup-js');	// Custombox JS
		wp_enqueue_script('paoc-public-js');			// Public JS
		paoc_pro_enqueue_script();

		// Print Inline Style
		echo "<style type='text/css'>{$atts['style']['inline']}</style>";

		// If Popup Preview is there
		if( $paoc_preview == 1 ) {
			$popup_content = $_POST['paoc_preview_form_data']['content'];
		} else {
			$popup_content = $popup_query[0]->post_content;
		}

		$secondary_content			= isset( $content['secondary_content'] ) ? $content['secondary_content'] : '';
		$atts['popup_content']		= paoc_pro_render_popup_content( $popup_content );
		$atts['secondary_content']	= paoc_pro_render_popup_content( $secondary_content );

		do_action( 'paoc_pro_before_popup_render', $atts );

		// Design File
		if( $display_type == 'bar' ) {

			paoc_pro_get_template( "{$display_type}/{$popup_goal}/{$atts['template']}.php", $atts );

		} else {

			paoc_pro_get_template( "{$display_type}/{$popup_goal}/{$atts['template']}.php", $atts, '', '', "designs/{$atts['template']}.php" );
		}

		do_action( 'paoc_pro_after_popup_render', $atts );

		/**
		 * Store global popup data in temp 2 variable after `do_shortcode`.
		 * Global popup data $paoc_popup_data is changed after `do_shortcode` bacause of popup within popup shortocde.
		 * This is used to avoid rendering same popup multiple times in DOM
		 */
		$paoc_global_temp_2	= $paoc_popup_data;

		// Temp 1 & Temp 2 array difference of key. Render only newly added popup in global popup data.
		$paoc_global_diff = array_diff_key( $paoc_global_temp_2, $paoc_global_temp_1 );

		// Render nested popup
		if( ! empty( $paoc_global_diff ) ) {
			$this->paoc_pro_render_nested_popup( $paoc_global_diff );
		}

		// Flush some global var
		$paoc_popup_post = '';
	}

	/**
	* Function to render a dummy image for adblocker detection after plugin public JS is loaded.
	* 
	* @since 2.0
	*/
	function paoc_pro_detect_adblocker() {

		global $paoc_adblocker;

		// If avance adblocker detection is there
		if( $paoc_adblocker == 2 ) {
			echo '<img class="paoc-pro-adblock-img" src="https://tpc.googlesyndication.com/simgad/7962139025367148577?" alt="" style="height:0;width:0;visibility:hidden;opacity:0;display:none;" onerror="paoc_pro_adblock_img_err()" onload="paoc_pro_adblock_img_suc()" />';
		}
	}

	/**
	 * Function to store popup clicks / views
	 * 
	 * @since 2.0
	 */
	function paoc_pro_store_popup_click_view() {

		$popup_id		= ! empty( $_POST['popup_id'] )		? paoc_pro_clean_number( $_POST['popup_id'] )		: 0;
		$campaign_id	= ! empty( $_POST['campaign_id'] )	? paoc_pro_clean_number( $_POST['campaign_id'] )	: 0;
		$flag			= ! empty( $_POST['type'] )			? paoc_pro_clean_number( $_POST['type'] )			: '';
		$popup_type		= ! empty( $_POST['popup_type'] )	? paoc_pro_clean( $_POST['popup_type'] )			: '';
		$result			= array(
							'success'	=> 0,
							'msg' 		=> esc_html__( 'Sorry, Something happened wrong.', 'popup-anything-on-click' )
						);

		// If Popup ID is not there
		if( empty( $popup_id ) ) {
			wp_send_json( $result );
		}

		// Insert popup clicks / views in tbl
		$record_id = paoc_pro_popup_ins_click_view( array(
														'popup_id'		=> $popup_id,
														'flag'			=> $flag,
														'campaign_id'	=> $campaign_id,
														'popup_type'	=> $popup_type,
													));

		// If data is inserted successfully then update post meta
		if( $record_id ) {

			// Save popup clicks / views to Post Meta
			paoc_pro_update_popup_click_view( $popup_id, $flag, $popup_type );

			$result	= array(
							'success'	=> 1,
							'msg'		=> esc_html__( 'Success', 'popup-anything-on-click' )
						);
		}

		wp_send_json( $result );
	}

	/**
	 * Function to process popup form submission
	 * 
	 * @since 2.0
	 */
	function paoc_pro_popup_form_submit() {

		global $wpdb, $current_user, $paoc_ref_page_url;

		$prefix				= PAOC_PRO_META_PREFIX;
		$flag				= 2;
		$form_store_data	= array();
		$ip_address			= paoc_pro_get_ip();
		$post_data			= isset( $_POST['form_data'] )			? parse_str( $_POST['form_data'], $form_data )		: '';
		$popup_id			= ! empty( $_POST['popup_id'] )			? paoc_pro_clean_number( $_POST['popup_id'] )		: 0;
		$campaign_id		= ! empty( $_POST['campaign_id'] )		? paoc_pro_clean_number( $_POST['campaign_id'] )	: 0;
		$popup_type			= ! empty( $_POST['popup_type'] )		? paoc_pro_clean( $_POST['popup_type'] )			: '';
		$paoc_ref_page_url	= ! empty( $_POST['ref_page_url'] )		? paoc_pro_clean_url( $_POST['ref_page_url'] )		: '';
		$store_no_data		= ! empty( $_POST['store_no_data'] )	? 1	: 0;
		$store_no_views		= ! empty( $_POST['store_no_views'] )	? 1	: 0;

		// Taking Notification Tab Data
		$notification		= get_post_meta( $popup_id, $prefix.'notification', true );
		$enable_email		= ! empty( $notification['enable_email'] )		? 1	: 0;
		$enable_user_email	= ! empty( $notification['enable_user_email'] )	? 1	: 0;

		// Taking some variable for `MailChimp`
		$integration		= get_post_meta( $popup_id, $prefix.'integration', true );
		$mc_enable			= ! empty( $integration['mc_enable'] )			? 1 : 0;
		$mc_lists			= ! empty( $integration['mc_lists'] )			? $integration['mc_lists']	: array();

		$submission_data	= array( 'name' => '', 'email' => '', 'phone' => '' );
		$result				= array(
								'success'	=> 0,
								'msg' 		=> esc_html__( 'Sorry, One or more fields have an error. Please check and try again.', 'popup-anything-on-click' )
							);

		// If Popup ID OR Form Data is not there
		if( empty( $popup_id ) || empty( $form_data ) ) {
			wp_send_json( $result );
		}

		// Taking some data
		$all_form_fields	= '';
		$popup_name			= get_the_title( $popup_id );
		$user_id			= isset( $current_user->ID ) ? $current_user->ID : 0;
		$content 			= get_post_meta( $popup_id, $prefix.'content', true );
		$form_fields		= isset( $content['form_fields'] )	? $content['form_fields'] : '';
		$form_data			= paoc_pro_clean( $form_data );

		if( $form_fields ) {
			foreach ($form_fields as $form_field_key => $form_field_data) {

				$require		= ! empty( $form_field_data['require'] ) 					? 1													: 0;
				$multiple		= ! empty( $form_field_data['multiple'] ) 					? 1													: 0;
				$type			= isset( $form_field_data['type'] ) 						? $form_field_data['type']							: 'text';
				$label			= isset( $form_field_data['label'] ) 						? $form_field_data['label']							: '';
				$submited_data	= isset( $form_data[ 'paoc_field_'.$form_field_key ] )		? $form_data[ 'paoc_field_'.$form_field_key ]		: '';
				$math_inp_data	= isset( $form_data[ 'paoc_math_input_'.$form_field_key ] )	? $form_data[ 'paoc_math_input_'.$form_field_key ]	: '';
				$math_data		= isset( $form_data[ 'paoc_math_'.$form_field_key ] )		? $form_data[ 'paoc_math_'.$form_field_key ]		: '';
				$submited_key	= "paoc_field_{$form_field_key}";

				// Sanitize based on type
				if( $type == 'textarea' ) {
					$submited_data = sanitize_textarea_field( $submited_data );
				} else if( $type == 'email' ) {
					$submited_data = sanitize_email( $submited_data );
				} else {
					$submited_data = paoc_pro_clean( $submited_data );
				}

				$form_store_data[ $form_field_key ] = $submited_data;

				/* Validation of Fields */
				if( ( $type == 'text' || $type == 'textarea' || $type == 'full_name' ) && $require && $submited_data == '' ) {
					$result['errors'][ $submited_key ] = esc_html__( 'Please enter required value.', 'popup-anything-on-click' );
				}

				/* Validation of Email Field */
				if( $type == 'email' && $require && ! is_email( $submited_data ) ) {
					$result['errors'][ $submited_key ] = esc_html__( 'Please enter valid email.', 'popup-anything-on-click' );
				}

				/* Validation of Number Field */
				if( $type == 'number' && $require && ! is_numeric( $submited_data ) ) {
					$result['errors'][ $submited_key ] = esc_html__( 'Please enter valid number.', 'popup-anything-on-click' );
				}

				/* Validation of Tel Field */
				if( $type == 'tel' && $require && ! preg_match('/^[0-9 .\-+]+$/i', $submited_data) ) {
					$result['errors'][ $submited_key ] = esc_html__( 'Please enter valid phone number.', 'popup-anything-on-click' );
				}

				/* Validation of Checkbox Field */
				if( $type == 'checkbox' && $require && empty( $submited_data ) ) {
					$result['errors'][ $submited_key ] = esc_html__( 'Validation error occured.', 'popup-anything-on-click' );
				}

				/* Validation of Dropdown Field */
				if( $type == 'dropdown' && $require && ( $submited_data == '' || ! array_filter( $submited_data ) ) ) {
					$result['errors'][ $submited_key ] = esc_html__('Select the dropdown value.', 'popup-anything-on-click');
				}

				/* Validation of Date Field */
				if( $type == 'date' && $require ) {
					
					$valid_date = date( 'Y-m-d', strtotime( $submited_data ) );

					if( $submited_data == '' || ! $valid_date ) {
						$result['errors'][ $submited_key ] = esc_html__( 'Please enter valid date.', 'popup-anything-on-click' );
					}
				}

				/* Validation of Captcha Field */
				if( $type == 'math_captcha' ) {

					// Check captcha calculation
					if ( empty( $math_inp_data ) ) {
						$result['errors'][ $submited_key ] = esc_html__( 'Please enter captcha value.', 'popup-anything-on-click' );
					} else {

						$paoc_token			= wp_salt('auth');
						$paoc_correct_ans 	= sha1( $paoc_token . $math_inp_data );
						$math_answer 		= isset( $math_data ) ? $math_data : '';

						if ( $math_answer != $paoc_correct_ans ) {
							$result['errors'][ $submited_key ] = esc_html__( 'Sorry, invalid captcha value.', 'popup-anything-on-click' );
						}
					}
				}

				// If type is `Dropdown`
				if( $type == 'dropdown' ) {

					// If dropdown is not 'multiple' selection to store value as a string
					if( empty( $multiple ) ) {

						$form_store_data[ $form_field_key ] = reset( $form_store_data[ $form_field_key ] );

					} else { // Else dropdown is 'multiple' selection to store value as a commaseprated

						$form_store_data[ $form_field_key ] = ( isset( $form_store_data[ $form_field_key ] ) && is_array( $form_store_data[ $form_field_key ] ) ) ? implode( ', ', $form_store_data[ $form_field_key ] ) : '';
					}
				}

				// Storing Submision Form Required Data
				if ( $type == 'email' && $submission_data['email'] == '' ) {
					$submission_data['email'] = $submited_data;
				}
				if ( $type == 'full_name' && $submission_data['name'] == '' ) {
					$submission_data['name'] = $submited_data;
				}
				if ( $type == 'tel' && $submission_data['phone'] == '' ) {
					$submission_data['phone'] = $submited_data;
				}

				$all_form_fields .= "{$label}: {$form_store_data[ $form_field_key ]}\n";

			} // End of foreach

			/* Insert Process When No Error */
			if( ! isset( $result['errors'] ) ) {

				// Check flag to store data in DB or not?
				if( $store_no_data == 0 ) {

					// Insert entry data
					$wpdb->insert( PAOC_PRO_ENTRY_TBL, 
										array(
											'popup_id'		=> $popup_id,
											'user_id'		=> $user_id,
											'popup_type'	=> $popup_type,
											'referral_link'	=> $paoc_ref_page_url,
											'ip_address'	=> $ip_address,
											'created_date'	=> current_time('mysql'),
											'modified_date'	=> current_time('mysql'),
										)
					);
					$insert_id = $wpdb->insert_id;
				}

				// If inserted ID is there
				if( $insert_id ) {

					foreach ($form_store_data as $form_store_key => $form_store_value) {

						if( $form_store_value == '' ) {
							continue;
						}

						$meta_key_type = $form_fields[ $form_store_key ]['type'];

						// Insert entry meta data
						$wpdb->insert( PAOC_PRO_ENTRY_META_TBL, 
											array(
												'entry_id'		=> $insert_id,
												'popup_id'		=> $popup_id,
												'meta_key'		=> $form_store_key,
												'meta_value'	=> $form_store_value,
												'meta_key_type'	=> $meta_key_type,
											)
						);
					}
				}

				// Check flag to store clicks in report DB or not?
				if( $store_no_views == 0 ) {

					// Insert form submission click in `PAOC_PRO_REPORT_TBL` table
					$record_id = paoc_pro_popup_ins_click_view( array(
																		'popup_id'		=> $popup_id,
																		'campaign_id'	=> $campaign_id,
																		'flag'			=> $flag,
																		'popup_type'	=> $popup_type,
																		'referral_link'	=> $paoc_ref_page_url,
																	));

					// Save popup clicks to Post Meta
					paoc_pro_update_popup_click_view( $popup_id, $flag, $popup_type );
				}

				// Check flag to store emails in MailChimp account or not?
				if( $mc_enable == 1 && $mc_lists ) {

					// Taking some variable
					$inte_field_arr	= array();
					$mailchimp_data	= new Paoc_Pro_MailChimp();

					foreach( $mc_lists as $mc_key => $mc_data ) {

						// Taking some variable
						$list_id			= isset( $mc_data['list_id'] )				? $mc_data['list_id']	: '';
						$fields				= isset( $mc_data['fields'] )				? $mc_data['fields']	: '';
						$mc_double_optin	= ! empty( $mc_data['mc_double_optin'] )	? 1 : 0;

						if( ! empty( $fields ) ) {
							foreach( $fields as $field_key => $field_val ) {
								$inte_field_arr[ $field_key ] = isset( $form_store_data[ $field_val ] ) ? $form_store_data[ $field_val ] : '';
							}
						}

						// If email form field is mapped then take it else take popup form email field
						if( ! empty( $inte_field_arr['EMAIL'] ) ) {
							$email_address = $inte_field_arr['EMAIL'];
						} else {
							$email_address = $submission_data['email'];
						}

						if( ! empty( $email_address ) ) {
							$mailchimp_data->update_data_in_list( $list_id, array(
																				'status'		=> $mc_double_optin,
																				'email'			=> $email_address,
																				'merge_fields'	=> $inte_field_arr,
																			));
						}

						if ( ! empty( $mc_data['mc_tags'] ) ) {
							$mailchimp_data->add_member_tag( $list_id, $email_address, $mc_data['mc_tags'] );
						}
					}
				}

				// Check flag to send mail to admin or not?
				if( $enable_email == 1 ) {

					// Taking some variable
					$email_subject	= isset( $notification['email_subject'] )	? $notification['email_subject']	: esc_html__( 'We have get your response.', 'popup-anything-on-click' );
					$email_to		= ! empty( $notification['email_to'] )		? $notification['email_to']			: get_option( 'admin_email' );
					$email_heading	= ! empty( $notification['email_heading'] )	? $notification['email_heading']	: '';
					$email_msg		= ! empty( $notification['email_msg'] )		? do_shortcode( wpautop( $notification['email_msg'] ) )	: esc_html__('Thanks for your time.', 'popup-anything-on-click');
					$email_subject	= str_replace( '{ID}', $popup_id, $email_subject );
					$email_msg		= str_replace( '{ID}', $popup_id, $email_msg );
					$email_msg		= str_replace( '{popup_name}', $popup_name, $email_msg );
					$email_msg		= str_replace( '{entry_id}', $insert_id, $email_msg );
					$email_msg		= str_replace( '{all_fields}', $all_form_fields, $email_msg );

					// Foreach for all form fields
					foreach( $form_store_data as $form_store_key => $form_store_val ) {

						$type = $form_fields[ $form_store_key ]['type'];

						// Captcha is skip
						if( $type == 'math_captcha' ) {
							continue;
						}

						$form_store_val	= is_array( $form_store_val ) ? implode( ',', $form_store_val ) : $form_store_val;
						$email_msg		= str_replace( '{field_'.$form_store_key.'}', $form_store_val, $email_msg );
					}

					$paoc_pro_email = new Paoc_Pro_Email();

					$paoc_pro_email->__set( 'heading', $email_heading );
					$paoc_pro_email->send( $email_to, $email_subject, $email_msg );
				}

				// Check flag to send mail to user or not?
				if( $enable_user_email == 1 && ! empty( $submission_data['email'] ) ) {

					// Taking some variable
					$user_email_subject	= isset( $notification['user_email_subject'] )		? $notification['user_email_subject']							: esc_html__( 'We have get your response.', 'popup-anything-on-click' );
					$user_email_heading	= ! empty( $notification['user_email_heading'] )	? $notification['user_email_heading']							: '';
					$user_email_msg		= ! empty( $notification['user_email_msg'] )		? do_shortcode( wpautop( $notification['user_email_msg'] ) )	: esc_html__('Thanks for your time.', 'popup-anything-on-click');
					$user_email_subject	= str_replace( '{name}', $submission_data['name'], $user_email_subject );
					$user_email_msg		= str_replace( '{ID}', $popup_id, $user_email_msg );
					$user_email_msg		= str_replace( '{popup_name}', $popup_name, $user_email_msg );
					$user_email_msg		= str_replace( '{entry_id}', $insert_id, $user_email_msg );
					$user_email_msg		= str_replace( '{all_fields}', $all_form_fields, $user_email_msg );

					// Foreach for all form fields
					foreach( $form_store_data as $form_store_key => $form_store_val ) {

						$type = $form_fields[ $form_store_key ]['type'];

						// Captcha is skip
						if( $type == 'math_captcha' ) {
							continue;
						}

						$form_store_val	= is_array( $form_store_val ) ? implode( ',', $form_store_val ) : $form_store_val;
						$user_email_msg	= str_replace( '{field_'.$form_store_key.'}', $form_store_val, $user_email_msg );
					}

					$paoc_pro_email = new Paoc_Pro_Email();

					$paoc_pro_email->__set( 'heading', $user_email_heading );
					$paoc_pro_email->send( $submission_data['email'], $user_email_subject, $user_email_msg );
				}

				$thanks_msg	= ! empty( $content['thanks_msg'] ) ? do_shortcode( wpautop( $content['thanks_msg'] ) ) : esc_html__('Thanks for your submission.', 'popup-anything-on-click');

				$result['success']		= 1;
				$result['msg']			= esc_html__( 'Success', 'popup-anything-on-click' );
				$result['thanks_msg']	= $thanks_msg;
			}
		}

		wp_send_json( $result );
	}

	/**
	 * Function to process popup form submission
	 * 
	 * @since 2.0.6
	 */
	function paoc_pro_process_popup_data() {

		$geo_location 	= ! empty( $_POST['geo_location'] ) ? 1 : 0;
		$result_data	= array();
		$result			= array(
								'success'	=> 0,
								'msg'		=> esc_js( __( 'Sorry, Something happened wrong.', 'popup-anything-on-click' ) ),
							);

		// Geolocation Data
		if( $geo_location ) {

			$geo_location_data	= paoc_pro_geolocate_ip( '', true, false );
			$geo_storage_time	= apply_filters( 'paoc_pro_geo_storage_time', 10 );
			$geo_storage_time	= paoc_pro_clean_number( $geo_storage_time, 10 );

			if( ! empty( $geo_location_data['country'] ) ) {

					$result_data['geo']['ip']		= $geo_location_data['ip_address'];
					$result_data['geo']['country']	= strtolower( $geo_location_data['country'] );
					$result_data['geo']['date']		= date_i18n( 'Y-m-d', strtotime("+{$geo_storage_time} days") );

					$result['success']	= 1;
					$result['data']		= $result_data;
					$result['msg']		= esc_js( __( 'Success', 'popup-anything-on-click' ) );
			}
		}

		wp_send_json( $result );
	}

	/**
	 * Function to get campaign popup ids and validate by same appereance
	 * else return original one
	 * 
	 * @since 2.0
	 */
	function paoc_pro_campaign_display_popup( $popup_id = 0 ) {

		// Taking some variable
		$campaign_popups	= paoc_pro_campaign_popups( $popup_id );
		$popup_ids			= ! empty( $campaign_popups['display_popup'] ) ? $campaign_popups['display_popup'] : $popup_id;

		return $popup_ids;
	}

	/**
	 * Function to handle module preview screen
	 * 
	 * @since 2.0
	 */
	function paoc_pro_render_popup_preview() {

		if( isset( $_GET['paoc_popup_preview'] ) && $_GET['paoc_popup_preview'] == 1 && ( isset( $_SERVER['HTTP_REFERER'] ) && (strpos($_SERVER['HTTP_REFERER'], 'post.php') !== false || strpos($_SERVER['HTTP_REFERER'], 'post-new.php') !== false) ) ) {
			include_once( PAOC_PRO_DIR . '/includes/preview/preview.php' );
			exit;
		}
	}
}

$paoc_pro_public = new PAOC_PRO_Public();