<?php
/**
 * Plugin generic functions file
 *
 * @package Popup Anything on Click Pro
 * @since 1.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Create table for form submission & popup report
 * 
 * @since 2.0
 */
function paoc_pro_create_tables() {

	global $wpdb, $charset_collate;

	require_once( ABSPATH . 'wp-admin/includes/upgrade.php' );

	$sql = "CREATE TABLE ".PAOC_PRO_ENTRY_TBL." (
		id bigint(20) NOT NULL AUTO_INCREMENT,
		popup_id bigint(20) NOT NULL,
		user_id bigint(20) NOT NULL,
		popup_type text NOT NULL,
		referral_link text NOT NULL,
		ip_address text NOT NULL,
		created_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
		modified_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
		PRIMARY KEY  (id)
	) $charset_collate;

	CREATE TABLE ".PAOC_PRO_ENTRY_META_TBL." (
		id bigint(20) NOT NULL AUTO_INCREMENT,
		entry_id bigint(20) NOT NULL,
		popup_id bigint(20) NOT NULL,
		meta_key bigint(20) NOT NULL,
		meta_value text NOT NULL,
		meta_key_type text NOT NULL,
		PRIMARY KEY  (id)
	) $charset_collate;

	CREATE TABLE ".PAOC_PRO_REPORT_TBL." (
		id bigint(20) NOT NULL AUTO_INCREMENT,
		popup_id bigint(20) NOT NULL,
		type int(11) NOT NULL,
		click_view bigint(20) NOT NULL,
		inline_click_view bigint(20) NOT NULL,
		campaign_id bigint(20) NOT NULL,
		created_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
		modified_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
		PRIMARY KEY  (id)
	) $charset_collate;

	CREATE TABLE ".PAOC_PRO_CAMPAIGN_TBL." (
		id bigint(20) NOT NULL AUTO_INCREMENT,
		enable text NOT NULL,
		title text NOT NULL,
		popup_ids text NOT NULL,
		popup_appear text NOT NULL,
		created_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
		modified_date datetime NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
		PRIMARY KEY  (id)
	) $charset_collate";

	dbDelta( $sql );
}

/**
 * Get Settings From Option Page
 * 
 * Handles to return all settings value
 * 
 * @package Popup Anything On Click Pro
 * @since 1.3
 */
function paoc_pro_get_settings() {
	
	$options	= get_option('paoc_pro_options');
	$settings	= is_array( $options ) ? $options : array();

	return $settings;
}

/**
 * Get an option
 * Looks to see if the specified setting exists, returns default if not
 * 
 * @package Popup Anything On Click Pro
 * @since 1.3
 */
function paoc_pro_get_option( $key = '', $default = false ) {

	global $paoc_pro_options;

	$value = ! empty( $paoc_pro_options[ $key ] ) ? $paoc_pro_options[ $key ] : $default;
	$value = apply_filters( 'paoc_pro_get_option', $value, $key, $default );

	return apply_filters( 'paoc_pro_get_option_' . $key, $value, $key, $default );
}

/**
 * Clean variables using sanitize_text_field. Arrays are cleaned recursively.
 * Non-scalar values are ignored.
 * 
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_clean( $var ) {
	if ( is_array( $var ) ) {
		return array_map( 'paoc_pro_clean', $var );
	} else {
		$data = is_scalar( $var ) ? sanitize_text_field( $var ) : $var;
		return wp_unslash($data);
	}
}

/**
 * Sanitize number value and return fallback value if it is blank
 * 
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_clean_number( $var, $fallback = null, $type = 'int' ) {

	$var = trim( $var );
	$var = is_numeric( $var ) ? $var : 0;

	if ( $type == 'number' ) {
		$data = intval( $var );
	} else if ( $type == 'abs' ) {
		$data = abs( $var );
	} else if ( $type == 'float' ) {
		$data = (float)$var;
	} else {
		$data = absint( $var );
	}

	return ( empty( $data ) && isset( $fallback ) ) ? $fallback : $data;
}

/**
 * Sanitize URL
 * 
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_clean_url( $url ) {
	return esc_url_raw( trim( $url ) );
}

/**
 * Sanitize Hex Color
 * 
 * @package Popup Anything on Click Pro
 * @since 1.2.2
 */
function paoc_pro_clean_color( $color, $fallback = null ) {

	if ( false === strpos( $color, 'rgba' ) ) {
		
		$data = sanitize_hex_color( $color );

	} else {

		$red	= 0;
		$green	= 0;
		$blue	= 0;
		$alpha	= 0.5;

		// By now we know the string is formatted as an rgba color so we need to further sanitize it.
		$color = str_replace( ' ', '', $color );
		sscanf( $color, 'rgba(%d,%d,%d,%f)', $red, $green, $blue, $alpha );
		$data = 'rgba('.$red.','.$green.','.$blue.','.$alpha.')';
	}

	return ( empty( $data ) && $fallback ) ? $fallback : $data;
}

/**
 * Allow Valid Html Tags
 * It will sanitize HTML (strip script and style tags)
 *
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_clean_html( $data = array() ) {

	if ( is_array( $data ) ) {

		$data = array_map( 'paoc_pro_clean_html', $data );

	} elseif ( is_string( $data ) ) {
		$data = trim( $data );
		$data = wp_filter_post_kses( $data );
	}

	return $data;
}

/**
 * Function to check email with comma separated values
 * 
 * @package Popup Anything on Click Pro
 * @since 2.0
 */
function paoc_pro_check_email( $emails ) {

	$correct_email	= array();
	$email_ids		= explode( ',', $emails );
	$email_ids		= paoc_pro_clean( $email_ids );

	foreach ( $email_ids as $email_key => $email_value ) {
		if( is_email( $email_value ) ) {
			$correct_email[] = $email_value;
		}
	}

	return implode( ', ', $correct_email );
}

/**
 * Sanitize Multiple HTML class
 * 
 * @package Popup Anything on Click Pro
 * @since 1.2.2
 */
function paoc_pro_sanitize_html_classes( $classes, $sep = " " ) {
	$return = "";

	if( $classes && ! is_array( $classes ) ) {
		$classes = explode( $sep, $classes );
	}

	if( ! empty( $classes ) ) {
		foreach( $classes as $class ){
			$return .= sanitize_html_class( $class ) . " ";
		}
		$return = trim( $return );
	}

	return $return;
}

/**
 * Function to add array after specific key
 * 
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_add_array( &$array, $value, $index, $from_last = false ) {

	if( is_array( $array ) && is_array( $value ) ) {

		if( $from_last ) {
			$total_count	= count( $array );
			$index			= ( ! empty( $total_count ) && ( $total_count > $index ) ) ? ( $total_count - $index ): $index;
		}
		
		$split_arr	= array_splice( $array, max( 0, $index ) );
		$array		= array_merge( $array, $value, $split_arr );
	}
	
	return $array;
}

/**
 * Function to get unique value number
 * 
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_get_unique() {
	static $unique = 0;
	$unique++;

	return $unique;
}

/**
 * Function to enqueue public script at last
 * 
 * @package Popup Anything on Click Pro
 * @since 2.0
 */
function paoc_pro_enqueue_script() {

	if( wp_script_is( 'paoc-public-js', 'enqueued' ) ) {

		// Dequeue Public JS
		wp_dequeue_script( 'paoc-public-js' );

		// Enqueue Public JS
		wp_enqueue_script('paoc-public-js');
	}
}

/**
 * Function to get popup effects
 * 
 * @package Popup Anything on Click Pro
 * @since 1.0
 */
function paoc_pro_popup_effects() {
	$popup_effect = array(
						'fadein'		=> __('Fadein', 'popup-anything-on-click'),
						'slide'			=> __('Slide', 'popup-anything-on-click'),
						'newspaper'		=> __('Newspaper', 'popup-anything-on-click'),
						'superscaled'	=> __('Super Scaled', 'popup-anything-on-click'),
						'corner'		=> __('Corner', 'popup-anything-on-click'),
						'scale'			=> __('Scale', 'popup-anything-on-click'),
						'slidetogether'	=> __('Slide Together', 'popup-anything-on-click'),
					);
	return apply_filters('paoc_pro_popup_effects', $popup_effect );
}

/**
 * Function get filter date options
 * 
 * @since 2.0
 */
function paoc_pro_report_date_options() {

	$date_options = array(
							'today'			=> __('Today', 'popup-anything-on-click'),
							'yesterday'		=> __('Yesterday','popup-anything-on-click'),
							'this_week'		=> __('This Week', 'popup-anything-on-click'),
							'last_week'		=> __('Last Week', 'popup-anything-on-click'),
							'last_30_days'	=> __('Last 30 Days', 'popup-anything-on-click'),
							'this_month'	=> __('This Month', 'popup-anything-on-click'),
							'last_month'	=> __('Last Month', 'popup-anything-on-click'),
							'this_quarter'	=> __('This Quarter', 'popup-anything-on-click'),
							'last_quarter'	=> __('Last Quarter', 'popup-anything-on-click'),
							'this_year'		=> __('This Year', 'popup-anything-on-click'),
							'last_year'		=> __('Last Year', 'popup-anything-on-click'),
							'other'			=> __('Custom', 'popup-anything-on-click'),
						);

	return apply_filters( 'paoc_pro_report_date_options', $date_options );
}

/**
 * Function to get popup appearance options
 * 
 * @since 2.0
 */
function paoc_pro_when_appear_options() {

	$popup_appear = array(
						'page_load'		=>	__('Page Load', 'popup-anything-on-click'),
						'inactivity'	=>	__('After X Second of Inactivity', 'popup-anything-on-click'),
						'scroll'		=>	__('When Page Scroll Down', 'popup-anything-on-click'),
						'scroll_up'		=>	__('When Page Scroll UP', 'popup-anything-on-click'),
						'exit'			=> 	__('Exit Intent', 'popup-anything-on-click'),
						'simple_link'	=> 	__('Simple Link', 'popup-anything-on-click'),
						'image'			=> 	__('Image Click', 'popup-anything-on-click'),
						'button'		=> 	__('Button Click', 'popup-anything-on-click'),
						'html_element'	=> 	__('HTML Element Click', 'popup-anything-on-click'),
					);

	return apply_filters('paoc_pro_when_appear_options', $popup_appear );
}

/**
 * When popup goal function
 * 
 * @since 2.0
 */
function paoc_pro_popup_goals() {

	$popup_goals = array(
						'announcement'	=>	array(
												'name'	=> __('Announcement', 'popup-anything-on-click'),
												'icon'	=> "dashicons dashicons-megaphone",
											),
						'email-lists'	=>	array(
												'name'	=> __('Collect Lead', 'popup-anything-on-click'),
												'icon'	=> "dashicons dashicons-email-alt",
											),
						'phone-calls'	=>	array(
												'name'	=> __('Phone Calls', 'popup-anything-on-click'),
												'icon'	=> "dashicons dashicons-phone",
											),
						'target-url'	=>	array(
												'name'	=> __('Target URL', 'popup-anything-on-click'),
												'icon'	=> "dashicons dashicons-admin-links",
											),
					);

	return apply_filters('paoc_pro_popup_goals', $popup_goals );
}

/**
 * When popup type function
 * 
 * @since 2.0
 */
function paoc_pro_popup_types() {

	$popup_types = array(
						'bar'				=>	array(
													'name'	=> __('Bar', 'popup-anything-on-click'),
													'icon'	=> "dashicons dashicons-schedule",
												),
						'modal'				=>	array(
													'name'	=> __('Modal Popup', 'popup-anything-on-click'),
													'icon'	=> "dashicons dashicons-admin-page",
												),
						'push-notification'	=>	array(
													'name'	=> __('Push Notification', 'popup-anything-on-click'),
													'icon'	=> "dashicons dashicons-admin-comments",
												),
						'slide-in'			=>	array(
													'name'	=> __('Slide In', 'popup-anything-on-click'),
													'icon'	=> "dashicons dashicons-align-right",
												),
					);

	return apply_filters('paoc_pro_popup_types', $popup_types );
}

/**
 * Function to get form field type options
 * 
 * @since 2.0
 */
function paoc_pro_form_field_type_options() {

	$field_type_options = array(
							'text'			=> array( 'type' => 'text', 'label' => __('Text', 'popup-anything-on-click') ),
							'email'			=> array( 'type' => 'email', 'label' => __('Email', 'popup-anything-on-click') ),
							'full_name'		=> array( 'type' => 'text', 'label' => __('Full Name (First Name & Last Name)', 'popup-anything-on-click') ),
							'tel'			=> array( 'type' => 'tel', 'label' => __('Phone Number', 'popup-anything-on-click') ),
							'number'		=> array( 'type' => 'number', 'label' => __('Number', 'popup-anything-on-click') ),
							'textarea'		=> array( 'type' => 'textarea', 'label' => __('Textarea', 'popup-anything-on-click') ),
							'checkbox'		=> array( 'type' => 'checkbox', 'label' => __('Checkbox', 'popup-anything-on-click') ),
							'dropdown'		=> array( 'type' => 'select', 'label' => __('Dropdown', 'popup-anything-on-click') ),
							'date'			=> array( 'type' => 'date', 'label' => __('Date Picker', 'popup-anything-on-click') ),
							'math_captcha'	=> array( 'type' => 'text', 'label' => __('Captcha', 'popup-anything-on-click') ),
						);
	return apply_filters('paoc_pro_form_field_type_options', $field_type_options );
}

/**
 * Function to get social options
 * 
 * @since 2.0
 */
function paoc_pro_social_options() {

	$social_options = array(
							'amazon'		=> array(
														'name' => __('Amazon', 'popup-anything-on-click'),
														'icon' => 'fa fa-amazon',
												),
							'bitbucket'		=> array(
														'name' => __('Bitbucket', 'popup-anything-on-click'),
														'icon' => 'fa fa-bitbucket',
												),
							'codepen'		=> array(
														'name' => __('Codepen', 'popup-anything-on-click'),
														'icon' => 'fa fa-codepen',
												),
							'delicious'		=> array(
														'name' => __('Delicious', 'popup-anything-on-click'),
														'icon' => 'fa fa-delicious',
												),
							'digg'			=> array(
														'name' => __('Digg', 'popup-anything-on-click'),
														'icon' => 'fa fa-digg',
												),
							'dribbble'		=> array(
														'name' => __('Dribbble', 'popup-anything-on-click'),
														'icon' => 'fa fa-dribbble',
												),
							'dropbox'		=> array(
														'name' => __('Dropbox', 'popup-anything-on-click'),
														'icon' => 'fa fa-dropbox',
												),
							'facebook'		=> array(
														'name' => __('Facebook', 'popup-anything-on-click'),
														'icon' => 'fa fa-facebook',
												),
							'flickr'		=> array(
														'name' => __('Flickr', 'popup-anything-on-click'),
														'icon' => 'fa fa-flickr',
												),
							'foursquare'	=> array(
														'name' => __('Foursquare', 'popup-anything-on-click'),
														'icon' => 'fa fa-foursquare',
												),
							'git'			=> array(
														'name' => __('Git', 'popup-anything-on-click'),
														'icon' => 'fa fa-git',
												),
							'github'		=> array(
														'name' => __('GitHub', 'popup-anything-on-click'),
														'icon' => 'fa fa-github',
												),
							'gitlab'		=> array(
														'name' => __('GitLab', 'popup-anything-on-click'),
														'icon' => 'fa fa-gitlab',
												),
							'google'		=> array(
														'name' => __('Google', 'popup-anything-on-click'),
														'icon' => 'fa fa-google',
												),
							'instagram'		 => array(
														'name' => __('Instagram', 'popup-anything-on-click'),
														'icon' => 'fa fa-instagram',
												),
							'jsfiddle'		=> array(
														'name' => __('JSFiddle', 'popup-anything-on-click'),
														'icon' => 'fa fa-jsfiddle',
												),
							'linkedin'		=> array(
														'name' => __('Linkedin', 'popup-anything-on-click'),
														'icon' => 'fa fa-linkedin',
												),
							'mail'			=> array(
														'name' => __('Email', 'popup-anything-on-click'),
														'icon' => 'fa fa-envelope-o',
												),
							'medium'		=> array(
														'name' => __('Medium', 'popup-anything-on-click'),
														'icon' => 'fa fa-medium',
												),
							'meetup'		=> array(
														'name' => __('Meetup', 'popup-anything-on-click'),
														'icon' => 'fa fa-meetup',
												),
							'mixcloud'		=> array(
														'name' => __('Mixcloud', 'popup-anything-on-click'),
														'icon' => 'fa fa-mixcloud',
												),
							'pinterest'		=> array(
														'name' => __('Pinterest', 'popup-anything-on-click'),
														'icon' => 'fa fa-pinterest',
												),
							'podcast'		=> array(
														'name' => __('Podcast', 'popup-anything-on-click'),
														'icon' => 'fa fa-podcast',
												),
							'quora'			=> array(
														'name' => __('Quora', 'popup-anything-on-click'),
														'icon' => 'fa fa-quora',
												),
							'reddit'		=> array(
														'name' => __('Reddit', 'popup-anything-on-click'),
														'icon' => 'fa fa-reddit',
												),
							'rss'			=> array(
														'name' => __('RSS', 'popup-anything-on-click'),
														'icon' => 'fa fa-rss',
												),
							'skype'			=> array(
														'name' => __('Skype', 'popup-anything-on-click'),
														'icon' => 'fa fa-skype',
												),
							'slack'			=> array(
														'name' => __('Slack', 'popup-anything-on-click'),
														'icon' => 'fa fa-slack',
												),
							'slideshare'	=> array(
														'name' => __('SlideShare', 'popup-anything-on-click'),
														'icon' => 'fa fa-slideshare',
												),
							'soundcloud'	=> array(
														'name' => __('SoundCloud', 'popup-anything-on-click'),
														'icon' => 'fa fa-soundcloud',
												),
							'spotify'		=> array(
														'name' => __('Spotify', 'popup-anything-on-click'),
														'icon' => 'fa fa-spotify',
												),
							'stack-overflow'=> array(
														'name' => __('Stack Overflow', 'popup-anything-on-click'),
														'icon' => 'fa fa-stack-overflow',
												),
							'tripadvisor'	=> array(
														'name' => __('Tripadvisor', 'popup-anything-on-click'),
														'icon' => 'fa fa-tripadvisor',
												),
							'tumblr'		=> array(
														'name' => __('Tumblr', 'popup-anything-on-click'),
														'icon' => 'fa fa-tumblr',
												),
							'twitch'		=> array(
														'name' => __('Twitch', 'popup-anything-on-click'),
														'icon' => 'fa fa-twitch',
												),
							'twitter'		=> array(
														'name' => __('Twitter', 'popup-anything-on-click'),
														'icon' => 'fa fa-twitter',
												),
							'vimeo'			=> array(
														'name' => __('Vimeo', 'popup-anything-on-click'),
														'icon' => 'fa fa-vimeo',
												),
							'vine'			=> array(
														'name' => __('Vine', 'popup-anything-on-click'),
														'icon' => 'fa fa-vine',
												),
							'vk'			=> array(
														'name' => __('VK', 'popup-anything-on-click'),
														'icon' => 'fa fa-vk',
												),
							'globe'			=> array(
														'name' => __('Website', 'popup-anything-on-click'),
														'icon' => 'fa fa-globe',
												),
							'whatsapp'		=> array(
														'name' => __('WhatsApp', 'popup-anything-on-click'),
														'icon' => 'fa fa-whatsapp',
												),
							'wikipedia-w'	=> array(
														'name' => __('Wikipedia', 'popup-anything-on-click'),
														'icon' => 'fa fa-wikipedia-w',
												),
							'wordpress'		=> array(
														'name' => __('WordPress', 'popup-anything-on-click'),
														'icon' => 'fa fa-wordpress',
												),
							'xing'			=> array(
														'name' => __('Xing', 'popup-anything-on-click'),
														'icon' => 'fa fa-xing',
												),
							'yelp'			=> array(
														'name' => __('Yelp', 'popup-anything-on-click'),
														'icon' => 'fa fa-yelp',
												),
							'youtube'		=> array(
														'name' => __('Youtube', 'popup-anything-on-click'),
														'icon' => 'fa fa-youtube',
												),
						);

	return apply_filters('paoc_pro_social_options', $social_options );
}

/**
 * Modal Popup templates function
 * 
 * @since 2.0
 */
function paoc_pro_bar_popup_designs() {

	$bar_popup_designs	= array(
								'design-1'	=>	__('Template 1', 'popup-anything-on-click'),
								'design-2'	=>	__('Template 2', 'popup-anything-on-click'),
								'design-3'	=>	__('Template 3', 'popup-anything-on-click'),
								'design-4'	=>	__('Template 4', 'popup-anything-on-click'),
								'design-5'	=>	__('Template 5', 'popup-anything-on-click'),
							);

	return apply_filters('paoc_pro_bar_popup_designs', $bar_popup_designs );
}

/**
 * Modal Popup templates function
 * 
 * @since 2.0
 */
function paoc_pro_popup_designs() {

	$popup_designs	= array(
								'design-1'	=>	__('Template 1', 'popup-anything-on-click'),
								'design-2'	=>	__('Template 2', 'popup-anything-on-click'),
								'design-3'	=>	__('Template 3', 'popup-anything-on-click'),
								'design-4'	=>	__('Template 4', 'popup-anything-on-click'),
								'design-5'	=>	__('Template 5', 'popup-anything-on-click'),
								'design-6'	=>	__('Template 6', 'popup-anything-on-click'),
								'design-7'	=>	__('Template 7', 'popup-anything-on-click'),
								'design-8'	=>	__('Template 8', 'popup-anything-on-click'),
								'design-9'	=>	__('Template 9', 'popup-anything-on-click'),
								'design-10'	=>	__('Template 10', 'popup-anything-on-click'),
							);

	return apply_filters('paoc_pro_popup_designs', $popup_designs );
}

/**
 * Function to get modal & push notification popup position options
 * 
 * @since 2.0
 */
function paoc_pro_mn_position_options() {

	$mn_position_option = array(
							'left-top'		=> __('Left Top', 'popup-anything-on-click'),
							'left-center'	=> __('Left Center', 'popup-anything-on-click'),
							'left-bottom'	=> __('Left Bottom', 'popup-anything-on-click'),
							'center-top'	=> __('Center Top', 'popup-anything-on-click'),
							'center-center'	=> __('Center Center', 'popup-anything-on-click'),
							'center-bottom'	=> __('Center Bottom', 'popup-anything-on-click'),
							'right-top'		=> __('Right Top', 'popup-anything-on-click'),
							'right-center'	=> __('Right Center', 'popup-anything-on-click'),
							'right-bottom'	=> __('Right Bottom', 'popup-anything-on-click'),
						);
	return apply_filters('paoc_pro_mn_position_options', $mn_position_option );
}

/**
 * Bar popup position
 * 
 * @since 2.0
 */
function paoc_pro_bar_position_options() {

	$bar_positions	= array(
						'top-push'		=>	__('Top Push', 'popup-anything-on-click'),
						'top'			=>	__('Top', 'popup-anything-on-click'),
						'bottom'		=>	__('Bottom', 'popup-anything-on-click'),
						'bottom-push'	=>	__('Bottom Push', 'popup-anything-on-click'),
					);

	return apply_filters('paoc_pro_bar_position_options', $bar_positions );
}

/**
 * Function to get display on options
 * 
 * @since 2.0
 */
function paoc_pro_display_on_options() {

	$display_on_opts = array(
						'every_device'	=>	__('Every Device', 'popup-anything-on-click'),
						'desktop_only'	=>	__('Desktop Only', 'popup-anything-on-click'),
						'mobile_only'	=>	__('Mobile Only', 'popup-anything-on-click'),
					);

	return apply_filters('paoc_pro_display_on_options', $display_on_opts );
}

/**
 * Function to get show for options
 * 
 * @since 2.0
 */
function paoc_pro_show_for_options() {

	$show_for_options = array(	
							'all'		=> __('Every One', 'popup-anything-on-click'),
							'guest'		=> __('Guest User', 'popup-anything-on-click'),
							'member'	=> __('Logged In User', 'popup-anything-on-click'),
						);
	return apply_filters('paoc_pro_show_for_options', $show_for_options );
}

/**
 * Function to get time options
 * 
 * @since 2.0
 */
function paoc_pro_time_options() {

	$time_options = array(	
					'day'		=> __('Days', 'popup-anything-on-click'),
					'hour'		=> __('Hours', 'popup-anything-on-click'),
					'minutes'	=> __('Minutes', 'popup-anything-on-click'),
				);
	return apply_filters( 'paoc_pro_time_options', $time_options );
}

/**
 * Function to get week days
 * 
 * @since 2.8
 */
function paoc_pro_week_days() {

	$day_opts = array(	
					'0'	=> esc_html__('Sunday', 'popup-anything-on-click'),
					'1'	=> esc_html__('Monday', 'popup-anything-on-click'),
					'2'	=> esc_html__('Tuesday', 'popup-anything-on-click'),
					'3'	=> esc_html__('Wednesday', 'popup-anything-on-click'),
					'4'	=> esc_html__('Thursday', 'popup-anything-on-click'),
					'5'	=> esc_html__('Friday', 'popup-anything-on-click'),
					'6'	=> esc_html__('Saturday', 'popup-anything-on-click'),
				);
	return apply_filters( 'paoc_pro_week_days', $day_opts );
}

/**
 * Function to check week days
 * 
 * @since 2.8
 */
function paoc_pro_check_week_days() {

	$day_opts = array(	
					'0'	=> 'sunday',
					'1'	=> 'monday',
					'2'	=> 'tuesday',
					'3'	=> 'wednesday',
					'4'	=> 'thursday',
					'5'	=> 'friday',
					'6'	=> 'saturday',
				);
	return apply_filters( 'paoc_pro_check_week_days', $day_opts );
}

/**
 * Function to display message, norice etc
 * 
 * @package Popup Anything on Click Pro
 * @since 2.0
 */
function paoc_pro_display_message( $type = 'update', $msg = '', $echo = 1 ) {

	switch ( $type ) {
		case 'reset':
			$msg = ! empty( $msg ) ? $msg : __( 'All settings reset successfully.', 'popup-anything-on-click');
			$msg_html = '<div id="message" class="updated notice notice-success is-dismissible">
							<p><strong>' . $msg . '</strong></p>
						</div>';
			break;

		case 'error':
			$msg = ! empty( $msg ) ? $msg : __( 'Sorry, Something happened wrong.', 'popup-anything-on-click');
			$msg_html = '<div id="message" class="error notice is-dismissible">
							<p><strong>' . $msg . '</strong></p>
						</div>';
			break;

		default:
			$msg = ! empty( $msg ) ? $msg : __('Your changes saved successfully.', 'popup-anything-on-click');
			$msg_html = '<div id="message" class="updated notice notice-success is-dismissible">
							<p><strong>'. $msg .'</strong></p>
						</div>';
			break;
	}

	if( $echo ) {
		echo wp_kses_post( $msg_html );
	} else {
		return wp_kses_post( $msg_html );
	}
}

/**
 * Function to get registered post types
 * 
 * @package Popup Anything on Click Pro
 * @since 2.0
 */
function paoc_pro_get_post_types( $args = array(), $exclude_post = array() ) {     

	$post_types 		= array();
	$args       		= ( ! empty( $args ) && is_array( $args ) ) ? $args : array( 'public' => true );
	$default_post_types = get_post_types( $args, 'name' );
	$exclude_post 		= ! empty( $exclude_post ) ? (array) $exclude_post : array();

	if( ! empty( $default_post_types ) ) {
		foreach ($default_post_types as $post_type_key => $post_data) {
			if( ! in_array( $post_type_key, $exclude_post ) ) {
				$post_types[$post_type_key] = $post_data->label;
			}
		}
	}

	return apply_filters('paoc_pro_get_post_types', $post_types );
}

/**
 * Function to display location.
 * 
 * @package Popup Anything on Click Pro
 * @since 2.0
 */
function paoc_pro_display_locations( $type = 'all', $all = true, $exclude = array() ) {

	$locations		= array();
	$exclude		= array_merge( array('attachment', 'revision', 'nav_menu_item'), $exclude);
	$all_post_types	= paoc_pro_get_post_types();
	$post_types		= array();

	foreach ( $all_post_types as $post_type => $post_data ) {
		if( $all ) {
			$type_label = __( 'All', 'popup-anything-on-click' ) .' '. $post_data;
		} else {
			$type_label = $post_data;
		}

		$locations[ $post_type ] = $type_label;
	}

	if ( 'global' != $type ) {
		
		$glocations = array(
			'is_front_page'	=> __( 'Front Page', 'popup-anything-on-click' ),
			'is_search'		=> __( 'Search Results', 'popup-anything-on-click' ),
			'is_404'		=> __( '404 Error Page', 'popup-anything-on-click' ),
			'is_archive'	=> __( 'All Archives', 'popup-anything-on-click' ),
			'all'			=> __( 'Whole Site', 'popup-anything-on-click' ),
		);

		$locations = array_merge( $locations, $glocations );	
	}

	// Exclude some post type or location
	if( ! empty( $exclude ) ) {
		foreach ($exclude as $location_key) {
			unset( $locations[ $location_key ] );
		}
	}

	return $locations;
}

/**
 * Function to get popup appear meta on suggestion type
 * 
 * @package Popup Anything on Click Pro
 * @since 2.0
 */
function paoc_pro_sugg_meta_data( $appear_meta ) {

	$meta_data = array();

	// Page_load, Scroll, Inactivity appear meta
	if( $appear_meta == 'welcome' ) {
		$meta_data	= array(
							'relation' => 'OR',
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'page_load',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'scroll',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'scroll_up',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'inactivity',
							)
						);
	}

	// Exit appear meta
	if( $appear_meta == 'exit' ) {
		$meta_data	= array(
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'exit',
							)
						);
	}

	// General Popup appear meta
	if( $appear_meta == 'general' ) {
		$meta_data	= array(
							'relation' => 'OR',
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'page_load',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'scroll',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'scroll_up',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'inactivity',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'exit',
							),
							array(
								'key'	=> '_aoc_popup_appear',
								'value'	=> 'html_element',
							),
						);
	}

	return apply_filters( 'paoc_pro_sugg_meta_data', $meta_data );
}

/**
 * Little Hack to avoid 503 error when too long URL is generate after multiple attemps at WP List Table Search
 * 
 * @since 2.0
 */
function paoc_pro_avoid_long_url() {
	$_SERVER['REQUEST_URI'] = remove_query_arg( '_wp_http_referer', $_SERVER['REQUEST_URI'] );
}

/**
 * Function to get post type posts
 * 
 * @since 2.0
 */
function paoc_pro_get_posts( $post_type = PAOC_PRO_POST_TYPE ) {

	$post_args = array(
		'post_type' 		=> $post_type,
		'post_status' 		=> 'publish',
		'posts_per_page' 	=> -1,
		'order'				=> 'DESC',
		'orderby' 			=> 'date',
	);

	$paoc_posts = get_posts( $post_args );

	return apply_filters('paoc_pro_get_posts', $paoc_posts, $post_args );
}

/**
 * Function to get date range query
 * 
 * @since 2.0
 */
function paoc_pro_date_range_sql( $date_selected ) {

	$dates 	= '';

	switch ( $date_selected ) {
		case 'today':
			$today				= date( 'Y-m-d', current_time('timestamp') );

			$dates				= " AND DATE(created_date) = '{$today}'";
			break;

		case 'yesterday':
			$yesterday			= date( 'Y-m-d', strtotime('-1 days') );

			$dates				= " AND DATE(created_date) = '{$yesterday}'";
			break;

		case 'this_week':
		case 'last_week':
			$base_time			= ( $date_selected === 'this_week' ) ? current_time( 'mysql' ) : date( 'Y-m-d h:i:s', current_time( 'timestamp' ) - WEEK_IN_SECONDS );
			$last_week			= get_weekstartend( $base_time, get_option( 'start_of_week' ) );
			$lweek_start		= date( 'Y-m-d', $last_week['start'] );
			$lweek_end			= date( 'Y-m-d', $last_week['end'] );

			$dates				= " AND DATE(created_date) BETWEEN '{$lweek_start}' AND '{$lweek_end}'";
			break;

		case 'last_30_days' :
			$start_dates		= date( 'Y-m-d', current_time('timestamp') );
			$end_dates			= date( 'Y-m-d', strtotime('-29 days') );

			$dates				= " AND DATE(created_date) BETWEEN '{$end_dates}' AND '{$start_dates}'";

			break;

		case 'this_month':
			$this_month_start	= date('Y-m-01');
			$this_month_end		= date('Y-m-t');

			$dates				= " AND DATE(created_date) BETWEEN '{$this_month_start}' AND '{$this_month_end}'";
			break;

		case 'last_month':
			$month_start		= new DateTime("first day of last month");
			$month_end			= new DateTime("last day of last month");
			$last_month_start	= $month_start->format('Y-m-d');
			$last_month_end		= $month_end->format('Y-m-d');

			$dates				= " AND DATE(created_date) BETWEEN '{$last_month_start}' AND '{$last_month_end}'";
			break;

		case 'this_quarter':
			$current_month		= date('n', current_time( 'timestamp' ));

			if($current_month <= 3) {

				$this_quarter_start	= date('Y-01-01');
				$this_quarter_end	= date('Y-03-t');
			}

			else if($current_month <= 6) {

				$this_quarter_start	= date('Y-04-01');
				$this_quarter_end	= date('Y-06-t');
			}

			else if($current_month <= 9) {

				$this_quarter_start	= date('Y-07-01');
				$this_quarter_end	= date('Y-09-t');
			}

			else {

				$this_quarter_start	= date('Y-10-01');
				$this_quarter_end	= date('Y-12-t');
			}

			$dates				= " AND DATE(created_date) BETWEEN '{$this_quarter_start}' AND '{$this_quarter_end}'";
			break;

		case 'last_quarter':
			$current_month		= date('n');

			if($current_month <= 3) {

				$last_year			= date('Y')-1;
				$last_quarter_start	= date( $last_year.'-10-01' );
				$last_quarter_end	= date( $last_year.'-12-t' );
			}

			else if($current_month <= 6) {

				$last_quarter_start	= date('Y-01-01');
				$last_quarter_end	= date('Y-03-t');
			}

			else if($current_month <= 9) {

				$last_quarter_start	= date('Y-04-01');
				$last_quarter_end	= date('Y-06-t');
			}

			else {

				$last_quarter_start	= date('Y-07-01');
				$last_quarter_end	= date('Y-09-t');
			}

			$dates				= " AND DATE(created_date) BETWEEN '{$last_quarter_start}' AND '{$last_quarter_end}'";
			break;

		case 'this_year':
			$this_year_start	= date('Y-01-01');
			$this_year_end		= date('Y-12-t');

			$dates				= " AND DATE(created_date) BETWEEN '{$this_year_start}' AND '{$this_year_end}'";
			break;

		case 'last_year':
			$last_year			= date('Y')-1;
			$last_year_start	= date( $last_year.'-01-01' );
			$last_year_end		= date( $last_year.'-12-t' );

			$dates				= " AND DATE(created_date) BETWEEN '{$last_year_start}' AND '{$last_year_end}'";
			break;

		case 'other':
			$from_date			= isset( $_GET['from_date'] ) 	? date( 'Y-m-d', strtotime( $_GET['from_date'] ) ) 	: '';
			$to_date			= isset( $_GET['to_date'] ) 	? date( 'Y-m-d', strtotime( $_GET['to_date'] ) ) 	: '';

			$dates				= " AND DATE(created_date) BETWEEN '{$from_date}' AND '{$to_date}'";
			break;
	}

	return $dates;
}

/**
 * Function to get integrations information
 * 
 * @since 2.0
 */
function paoc_pro_integration_info( $integration = '', $type = 'lists' ) {

	global $paoc_intgs_info;

	$paoc_intgs_info = ! empty( $paoc_intgs_info ) ? $paoc_intgs_info : get_option( 'paoc_intgs_info' );

	/* Backward Compatibility (When we have only one integration) */
	if ( empty( $paoc_intgs_info ) ) {

		$mc_info = get_option('paoc_mc_info');

		if ( ! empty( $mc_info ) ) {
			return isset( $mc_info[$type] ) ? $mc_info[$type] : $mc_info;
		}
	}

	if ( $integration && $type ) {
		
		$intgs_info = isset( $paoc_intgs_info[$integration][$type] ) ? $paoc_intgs_info[$integration][$type] : array();

	} else {

		$intgs_info = $paoc_intgs_info;
	}

	return $intgs_info;
}

/**
 * Function to User IP address
 * 
 * @since 2.0
 */
function paoc_pro_get_ip() {

	if ( isset( $_SERVER['HTTP_X_REAL_IP'] ) ) { // WPCS: input var ok, CSRF ok.
		return sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_REAL_IP'] ) );  // WPCS: input var ok, CSRF ok.
	} elseif ( isset( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) { // WPCS: input var ok, CSRF ok.
		// Proxy servers can send through this header like this: X-Forwarded-For: client1, proxy1, proxy2
		// Make sure we always only send through the first IP in the list which should always be the client IP.
		return (string) rest_is_ip_address( trim( current( preg_split( '/,/', sanitize_text_field( wp_unslash( $_SERVER['HTTP_X_FORWARDED_FOR'] ) ) ) ) ) ); // WPCS: input var ok, CSRF ok.
	} elseif ( isset( $_SERVER['REMOTE_ADDR'] ) ) { // @codingStandardsIgnoreLine
		return sanitize_text_field( wp_unslash( $_SERVER['REMOTE_ADDR'] ) ); // @codingStandardsIgnoreLine
	}
	return '';
}

/**
 * Function to return wheather popup is active or not.
 * 
 * @since 2.0
 */
function paoc_pro_check_active( $glob_locs = array() ) {

	global $post, $paoc_popup_active;

	$prefix 			= PAOC_PRO_META_PREFIX;
	$paoc_post_type		= isset( $post->post_type ) ? $post->post_type : '';
	$custom_location	= false;
	$paoc_popup_active	= false;

	// Whole Website
	if( ! empty( $glob_locs['all'] ) ) {
		$paoc_popup_active = true;
	}

	// Post Type Wise
	if( ! empty( $glob_locs[ $paoc_post_type ] ) && is_singular() ) {
		$paoc_popup_active = true;
	}

	// Checking custom locations
	if( is_search() ) {
		$custom_location = "is_search";
	} else if( is_404() ) {
		$custom_location = "is_404";
	} else if( is_archive() ) {
		$custom_location = "is_archive";
	} else if( is_front_page() ) {
		$custom_location = "is_front_page";
	}

	if( $custom_location && ! empty( $glob_locs[ $custom_location ] ) ) {
		$paoc_popup_active = true;
	}

	return $paoc_popup_active;
}

/**
 * Get post meta
 * If preview is there then get run time post meta
 * 
 * @since 2.0
 */
function paoc_pro_get_post_status( $post_id ) {

	global $paoc_preview;

	$post_status = get_post_status( $post_id );

	// If popup preview is there
	if( $paoc_preview && ! empty( $_POST['paoc_preview_form_data'] ) ) {
		$post_status = 'publish';
	}

	return $post_status;
}

/**
 * Get popup default meta data
 * 
 * @since 2.0
 */
function paoc_pro_popup_default_meta() {

	$prefix = PAOC_PRO_META_PREFIX;

	$default_meta = array(
		'content'			=> __('Primary Content – Primary Content Goes Here.', 'popup-anything-on-click'),
		$prefix.'behaviour'	=> array(
								'close_overlay'	=> 1,
							),
		$prefix.'content'	=> array(
									'main_heading'		=> __('Main Heading Goes Here', 'popup-anything-on-click'),
									'sub_heading'		=> __('Sub Heading Goes Here', 'popup-anything-on-click'),
									'cust_close_txt'	=> __('No, thank you. I do not want.', 'popup-anything-on-click'),
									'security_note'		=> __('100% secure your website.', 'popup-anything-on-click'),
									'target_url'		=> array(
															'btn1_text'	=> __('Yes, I want', 'popup-anything-on-click'),
															'btn1_link'	=> site_url(),
															'btn2_text'	=> __('No, I do not want', 'popup-anything-on-click'),
														),
									'phone_calls'		=> array(
															'btn_txt'	=> __('Call Now', 'popup-anything-on-click'),
															'phone_num'	=> '447911123456',
														),
								),
		$prefix.'design'	=> array(
								'popup_img'	=> PAOC_PRO_URL.'/assets/images/preview.png',
							),
	);

	return apply_filters( 'paoc_pro_popup_default_meta', $default_meta );
}

/**
 * Get post meta
 * If preview is there then get run time post meta
 * 
 * @since 2.0
 */
function paoc_pro_get_meta( $post_id, $meta_key, $flag = true ) {

	global $pagenow, $paoc_preview;

	// If popup preview is there
	if( $paoc_preview && ! empty( $_POST['paoc_preview_form_data'] ) ) {

		if( ! is_array( $_POST['paoc_preview_form_data'] ) ) {
			parse_str( $_POST['paoc_preview_form_data'], $form_data );
		} else {
			$form_data = $_POST['paoc_preview_form_data'];
		}

		$post_meta = isset( $form_data[ $meta_key ] ) ? stripslashes_deep( $form_data[ $meta_key ] ) : '';

	} else {

		$default_meta	= paoc_pro_popup_default_meta();

		$post_meta		= get_post_meta( $post_id, $meta_key, $flag );
		$post_meta		= ( $pagenow == 'post-new.php' && isset( $default_meta[ $meta_key ] ) ) ? $default_meta[ $meta_key ] : $post_meta;
	}

	return $post_meta;
}

/**
 * Popup Preview Data
 * Excluded some meta data like 'Advance' to avoid creating cookie and etc which stops popup to render.
 * 
 * @since 2.0
 */
function paoc_pro_preview_data( $post_data ) {

	$prefix			= PAOC_PRO_META_PREFIX;
	$show_credit	= isset( $post_data[ $prefix.'advance' ]['show_credit'] ) ? $post_data[ $prefix.'advance' ]['show_credit'] : '';

	unset( $post_data[ $prefix.'advance'] );

	$post_data[ $prefix.'advance' ]['show_credit']	= $show_credit;
	$post_data[ $prefix.'advance' ]['display_on']	= 'every_device';
	$post_data[ $prefix.'popup_appear' ]			= 'page_load';
	$post_data[ $prefix.'behaviour' ]['open_delay'] = '';
	$post_data[ $prefix.'behaviour']['disappear']	= '';

	// Set default preview data
	if ( ! empty( $_GET['paoc_default_popup_preview'] ) ) {

		// Content Tab Data
		$post_data['content']									= ( isset ( $post_data['content'] ) && $post_data['content'] != '' )																? $post_data['content']									: __('Primary Content - Lorem Ipsum is simply dummy text of the printing and typesetting industry.', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['main_heading']			= ( isset ( $post_data[ $prefix.'content' ]['main_heading'] ) && $post_data[ $prefix.'content' ]['main_heading'] != '' )			? $post_data[ $prefix.'content' ]['main_heading']		: __('Main Heading Goes Here', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['sub_heading']			= ( isset ( $post_data[ $prefix.'content' ]['sub_heading'] ) && $post_data[ $prefix.'content' ]['sub_heading'] != '' )				? $post_data[ $prefix.'content' ]['sub_heading']		: __('Sub Heading Goes Here', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['secondary_content']	= ( isset ( $post_data[ $prefix.'content' ]['secondary_content'] ) && $post_data[ $prefix.'content' ]['secondary_content'] != '' )	? $post_data[ $prefix.'content' ]['secondary_content']	: __('Secondary Content - Lorem Ipsum is simply dummy text of the printing and typesetting industry.', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['cust_close_txt']		= ( isset ( $post_data[ $prefix.'content' ]['cust_close_txt'] ) && $post_data[ $prefix.'content' ]['cust_close_txt'] != '' )		? $post_data[ $prefix.'content' ]['cust_close_txt']		: __('No, thank you. I do not want.', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['security_note']		= ( isset ( $post_data[ $prefix.'content' ]['security_note'] ) && $post_data[ $prefix.'content' ]['security_note'] != '' )			? $post_data[ $prefix.'content' ]['security_note']		: __('100% secure your website.', 'popup-anything-on-click');

		// Target URL
		$post_data[ $prefix.'content' ]['target_url']['btn1_text']	= ( isset ( $post_data[ $prefix.'content' ]['target_url']['btn1_text'] ) && $post_data[ $prefix.'content' ]['target_url']['btn1_text'] != '' )	? $post_data[ $prefix.'content' ]['target_url']['btn1_text']	: __('Yes, I want.', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['target_url']['btn1_link']	= ( isset ( $post_data[ $prefix.'content' ]['target_url']['btn1_link'] ) && $post_data[ $prefix.'content' ]['target_url']['btn1_link'] != '' )	? $post_data[ $prefix.'content' ]['target_url']['btn1_link']	: 'https://www.essentialplugin.com/pricing/';
		$post_data[ $prefix.'content' ]['target_url']['btn2_text']	= ( isset ( $post_data[ $prefix.'content' ]['target_url']['btn2_text'] ) && $post_data[ $prefix.'content' ]['target_url']['btn2_text'] != '' )	? $post_data[ $prefix.'content' ]['target_url']['btn2_text']	: __('No, I do not want', 'popup-anything-on-click');

		// Phone Calls
		$post_data[ $prefix.'content' ]['phone_calls']['btn_txt']	= ( isset ( $post_data[ $prefix.'content' ]['phone_calls']['btn_txt'] ) && $post_data[ $prefix.'content' ]['phone_calls']['btn_txt'] != '' )		? $post_data[ $prefix.'content' ]['phone_calls']['btn_txt']		: __('Call Now', 'popup-anything-on-click');
		$post_data[ $prefix.'content' ]['phone_calls']['phone_num']	= ( isset ( $post_data[ $prefix.'content' ]['phone_calls']['phone_num'] ) && $post_data[ $prefix.'content' ]['phone_calls']['phone_num'] != '' )	? $post_data[ $prefix.'content' ]['phone_calls']['phone_num']	: '447911123456';

		// Design Data
		$post_data[ $prefix.'design' ]['popup_img'] = ( isset ( $post_data[ $prefix.'design' ]['popup_img'] ) && $post_data[ $prefix.'design' ]['popup_img'] != '' ) ? $post_data[ $prefix.'design' ]['popup_img'] : PAOC_PRO_URL.'/assets/images/preview.png';

		// Social Tab Data
		$post_data[ $prefix.'social' ]['social_traffic'][0]['name'] = ( isset ( $post_data[ $prefix.'social' ]['social_traffic'][0]['link'] ) && $post_data[ $prefix.'social' ]['social_traffic'][0]['link'] != '' ) ? $post_data[ $prefix.'social' ]['social_traffic'][0]['name'] : 'facebook';
		$post_data[ $prefix.'social' ]['social_traffic'][0]['link'] = ( isset ( $post_data[ $prefix.'social' ]['social_traffic'][0]['link'] ) && $post_data[ $prefix.'social' ]['social_traffic'][0]['link'] != '' ) ? $post_data[ $prefix.'social' ]['social_traffic'][0]['link'] : '#';
		$post_data[ $prefix.'social' ]['social_traffic'][1]['name'] = ( isset ( $post_data[ $prefix.'social' ]['social_traffic'][1]['link'] ) && $post_data[ $prefix.'social' ]['social_traffic'][1]['link'] != '' ) ? $post_data[ $prefix.'social' ]['social_traffic'][1]['name'] : 'twitter';
		$post_data[ $prefix.'social' ]['social_traffic'][1]['link'] = ( isset ( $post_data[ $prefix.'social' ]['social_traffic'][1]['link'] ) && $post_data[ $prefix.'social' ]['social_traffic'][1]['link'] != '' ) ? $post_data[ $prefix.'social' ]['social_traffic'][1]['link'] : '#';
		$post_data[ $prefix.'social' ]['social_traffic'][2]['name'] = ( isset ( $post_data[ $prefix.'social' ]['social_traffic'][2]['link'] ) && $post_data[ $prefix.'social' ]['social_traffic'][2]['link'] != '' ) ? $post_data[ $prefix.'social' ]['social_traffic'][2]['name'] : 'linkedin';
		$post_data[ $prefix.'social' ]['social_traffic'][2]['link'] = ( isset ( $post_data[ $prefix.'social' ]['social_traffic'][2]['link'] ) && $post_data[ $prefix.'social' ]['social_traffic'][2]['link'] != '' ) ? $post_data[ $prefix.'social' ]['social_traffic'][2]['link'] : '#';
	}

	return $post_data;
}

/**
 * Function to get current page URL
 * 
 * @since 2.0
 */
function paoc_pro_get_current_page_url( $args = array() ) {

	$curent_page_url = is_ssl() ? 'https://' : 'http://';

	// Check server port is not 80
	if ( $_SERVER["SERVER_PORT"] != "80" ) {
		$curent_page_url .= $_SERVER["SERVER_NAME"].":".$_SERVER["SERVER_PORT"].$_SERVER["REQUEST_URI"];
	} else {
		$curent_page_url .= $_SERVER["SERVER_NAME"].$_SERVER["REQUEST_URI"];
	}

	// Remove Query Args
	if( isset( $args['remove_args'] ) ) {
		$curent_page_url = remove_query_arg( $args['remove_args'], $curent_page_url );
	}

	return apply_filters( 'paoc_pro_get_current_page_url', $curent_page_url );
}

/**
 * Function to get social data of plugin
 * 
 * @since 2.0
 */
function paoc_pro_get_social_data( $social_arr = array() ) {

	$result			= array();
	$social_options	= paoc_pro_social_options();
	$social_arr		= ! is_array( $social_arr ) ? (array)$social_arr : $social_arr;

	if( ! empty( $social_arr ) ) {
		foreach ($social_arr as $social_key => $social_data) {

			if( empty( $social_data['name'] ) || empty( $social_data['link'] ) ) {
				continue;
			}

			$result[ $social_key ]			= $social_data;
			$result[ $social_key ]['title'] = ! empty( $social_options[ $social_data['name'] ]['name'] ) ? $social_options[ $social_data['name'] ]['name'] : ucfirst( $social_data['name'] );
			$result[ $social_key ]['icon']	= ! empty( $social_options[ $social_data['name'] ]['icon'] ) ? $social_options[ $social_data['name'] ]['icon'] : '';
		}
	}

	return $result;
}

/**
 * Function to insert popup clicks / views
 * 
 * @since 2.0
 */
function paoc_pro_popup_ins_click_view( $popup_args = array() ) {

	global $wpdb;

	// If no popup id is set
	if( empty( $popup_args['popup_id'] ) ) {
		return false;
	}

	// Taking some variable
	$today_date		= date( 'Y-m-d', current_time('timestamp') );
	$flag			= isset( $popup_args['flag'] )			? $popup_args['flag']		: 1;
	$popup_type		= isset( $popup_args['popup_type'] )	? $popup_args['popup_type']	: '';
	$popup_id		= $popup_args['popup_id'];
	$campaign_id	= $popup_args['campaign_id'];

	// Get if popup view entry is exist for the day
	$popup_report = $wpdb->get_row( "SELECT * FROM `".PAOC_PRO_REPORT_TBL."` WHERE popup_id = '{$popup_id}' AND `type` = '{$flag}' AND DATE_FORMAT(created_date, '%Y-%m-%d') = '{$today_date}' LIMIT 1", ARRAY_A );

	// Update entry
	if( $popup_report ) {

		// If popup type is inline to store inline click_view
		if( $popup_type == 'inline' ) {

			$wpdb->query( "UPDATE ".PAOC_PRO_REPORT_TBL." SET `inline_click_view`= `inline_click_view` + 1, `campaign_id`='".$campaign_id."', `modified_date` = '".current_time('mysql')."' WHERE `id` = '{$popup_report['id']}'" );

		} else {

			$wpdb->query( "UPDATE ".PAOC_PRO_REPORT_TBL." SET `click_view`= `click_view` + 1, `campaign_id`='".$campaign_id."', `modified_date` = '".current_time('mysql')."' WHERE `id` = '{$popup_report['id']}'" );
		}

		$record_id = $popup_report['id'];

	} else { // Insert Entry

		// Insert popup data array
		$insert_data_arr = array(
			'popup_id'		=> $popup_id,
			'type'			=> $flag,
			'campaign_id'	=> $campaign_id,
			'created_date'	=> current_time('mysql'),
			'modified_date'	=> current_time('mysql'),
		);

		// If popup type is inline to store inline click_view
		if( $popup_type == 'inline' ) {

			$insert_data_arr['inline_click_view'] = 1;

		} else {

			$insert_data_arr['click_view'] = 1;
		}

		// Insert some entry
		$wpdb->insert( PAOC_PRO_REPORT_TBL, $insert_data_arr );

		$record_id = $wpdb->insert_id;
	}

	return $record_id;
}

/**
 * Function to update popup meta views or clicks
 * $flag = 1 - For Popup View
 * $flag = 2 - For Popup Click
 * 
 * @since 2.0
 */
function paoc_pro_update_popup_click_view( $popup_id = 0, $flag = 1, $popup_type = 'normal' ) {

	$result = false;

	if( empty( $popup_id ) || ! isset( $flag ) ) {
		return $result;
	}

	$prefix = PAOC_PRO_META_PREFIX;

	// Views
	if( $flag == 1 ) {

		$popup_views	= get_post_meta( $popup_id, $prefix.'popup_views', true );
		$popup_views	= ( ! empty( $popup_views ) && is_array( $popup_views ) ) ? $popup_views : array('inline' => 0, 'normal' => 0);

		// If popup type is inline to store view
		if( $popup_type == 'inline' ) {

			$popup_views['inline'] = ( $popup_views['inline'] ) ? ($popup_views['inline'] + 1) : 1;

		} else { // Else popup type is normal to store view

			$popup_views['normal'] = ( $popup_views['normal'] ) ? ($popup_views['normal'] + 1) : 1;
		}

		return update_post_meta( $popup_id, $prefix.'popup_views', $popup_views );

	} else if( $flag == 2 ) { // Clicks

		$popup_clicks	= get_post_meta( $popup_id, $prefix.'popup_clicks', true );
		$popup_clicks	= ( ! empty( $popup_clicks ) && is_array( $popup_clicks ) ) ? $popup_clicks : array('inline' => 0, 'normal' => 0);

		// If popup type is inline to store click
		if( $popup_type == 'inline' ) {

			$popup_clicks['inline'] = ( $popup_clicks['inline'] ) ? ($popup_clicks['inline'] + 1) : 1;

		} else { // Else popup type is normal to store click

			$popup_clicks['normal'] = ( $popup_clicks['normal'] ) ? ($popup_clicks['normal'] + 1) : 1;
		}

		return update_post_meta( $popup_id, $prefix.'popup_clicks', $popup_clicks );
	}

	return $result;
}

/**
 * Convert color hex value to rgb format
 * 
 * @since 2.0
 */
function paoc_pro_hex2rgb( $hex = '', $format = 'string' ) {

	if( empty( $hex ) ) return false;

	$hex = str_replace("#", "", $hex);

	if( strlen( $hex ) == 3 ) {
		$r = hexdec( substr($hex, 0, 1).substr($hex, 0, 1) );
		$g = hexdec( substr($hex, 1, 1).substr($hex, 1, 1) );
		$b = hexdec( substr($hex, 2, 1).substr($hex, 2, 1) );
	} else {
		$r = hexdec( substr($hex, 0, 2) );
		$g = hexdec( substr($hex, 2, 2) );
		$b = hexdec( substr($hex, 4, 2) );
	}

	$rgb = array($r, $g, $b);

	if( $format == 'string' ) {
		$rgb = implode(",", $rgb);
	}

	return $rgb;
}

/**
 * Function to render popup content.
 * An alternate solution of apply_filter('the_content')
 *
 * Prioritize the function in a same order apply_filter('the_content') wp-includes/default-filters.php
 * 
 * @since 2.0
 */
function paoc_pro_render_popup_content( $popup_content = '', $only_allowed_html = FALSE ) {

	if ( empty( $popup_content ) ) {
		return false;
	}

	global $wp_embed;

	$popup_content		= $wp_embed->run_shortcode( $popup_content );
	$popup_content		= $wp_embed->autoembed( $popup_content );
	$popup_content		= wptexturize( $popup_content );
	$popup_content		= wpautop( $popup_content );
	$popup_content		= shortcode_unautop( $popup_content );

	// Since Version 5.5.0
	if ( function_exists('wp_filter_content_tags') ) {
		$popup_content = wp_filter_content_tags( $popup_content );
	}

	// Since Version 5.7.0
	if ( function_exists( 'wp_replace_insecure_home_url' ) ) {
		$popup_content = wp_replace_insecure_home_url( $popup_content );
	}

	$popup_content	= do_shortcode( $popup_content );
	$popup_content	= convert_smilies( $popup_content );
	$popup_content	= str_replace( ']]>', ']]&gt;', $popup_content );

	// Escaping with allowed Post HTML + IFRAME tag
	if ( $only_allowed_html ) {
		$popup_content = wp_kses( $popup_content, array_merge(
															array(
																'iframe' => array(
																	'src'				=> true,
																	'style'				=> true,
																	'id'				=> true,
																	'class'				=> true,
																	'height'			=> true,
																	'width'				=> true,
																	'title'				=> true,
																	'aria-describedby'	=> true,
																	'aria-details'		=> true,
																	'aria-label'		=> true,
																	'aria-labelledby'	=> true,
																	'aria-hidden' 		=> true,
																	'frameborder'		=> true,
																	'allowfullscreen'	=> true,
																	'loading'			=> true,
																	'allow'				=> true,
																	'data-*'			=> true,
																)
															),
															wp_kses_allowed_html( 'post' )
														) );
	}

	return apply_filters( 'paoc_pro_popup_content', $popup_content );
}

/*
*/
function paoc_pro_generate_modal_popup( $identifier = '' ) {

	// Popup Tags File
	include( PAOC_PRO_DIR . '/includes/admin/popup/popup-html.php' );
	$identifier = '';
}

/**
 * Function to return `Start Date` & `End Date` for Recurring Timer
 * 
 * @since 2.8
 */
function paoc_pro_recurring_dates( $popup_id = 0 ) {

	global $paoc_advance_sett;

	// Taking some variable
	$results	= array(
					'start_time'	=> '',
					'end_time'		=> '',
				);

	// Return If `Timer Mode` is not `Recurring Timer`
	if( empty( $popup_id ) ) {
		return $results;
	}

	// Taking some variable
	$current_time	= current_time( 'timestamp' );
	$schedule_mode	= ! empty( $paoc_advance_sett['schedule_mode'] ) ? $paoc_advance_sett['schedule_mode'] : 'default';

	// If `Schedule Type` is not `Recurring`
	if( $schedule_mode != 'recurring' ) {
		return $results;
	}

	$recurring_type		= ! empty( $paoc_advance_sett['recurring_type'] )	? $paoc_advance_sett['recurring_type']		: 'daily';
	$recur_start_time	= ! empty( $paoc_advance_sett['recur_start_time'] )	? $paoc_advance_sett['recur_start_time']	: '00:00:00';
	$recur_end_time		= ! empty( $paoc_advance_sett['recur_end_time'] )	? $paoc_advance_sett['recur_end_time']		: '23:59:59';
	$recur_start_time	= strtotime( $recur_start_time );
	$recur_end_time		= strtotime( $recur_end_time );

	if( $recurring_type == 'daily' ) {

		$results['start_time']	= date_i18n( 'Y-m-d H:i:s', $recur_start_time );
		$results['end_time']	= date_i18n( 'Y-m-d H:i:s', $recur_end_time );

	} else if( $recurring_type == 'weekly' ) { // If `Recurring Type` is `Weekly`

		// Taking some variable
		$rang_arr			= array();
		$week_days			= paoc_pro_check_week_days();
		$recur_week_start	= isset( $paoc_advance_sett['recur_week_start'] )	? $paoc_advance_sett['recur_week_start']	: 1;
		$recur_week_end		= isset( $paoc_advance_sett['recur_week_end'] )		? $paoc_advance_sett['recur_week_end']		: 0;
		$curr_day_of_week	= strtolower( current_time( 'l' ) );				// Current Day Name
		$curr_day_number	= date_i18n( "w", strtotime( $curr_day_of_week ) );	// Current Day Number
		$num				= $recur_week_start;

		for ($i = 0; $i <= 6; $i++) {

			// If check number not in rang array
			if( ! in_array( $num, $rang_arr ) ) {
				$rang_arr[] = $num;
			}

			// If check `Number` is equal to `Week End` number
			if( $num == $recur_week_end ) {
				break;
			}

			$num++; // Number increment

			// If check `Number` is greater then `6`
			if( $num > 6 ) {
				$num = 0;
			}
		}

		// If check current day is there in `rang` array
		if( in_array( $curr_day_number, $rang_arr ) ) {

			// Get start date time
			$get_sday_name			= $week_days[ $recur_week_start ];
			$last_start_day			= strtotime( 'last '.$get_sday_name, $current_time );
			$week_start_day			= ( $curr_day_of_week != $get_sday_name ) ? $last_start_day : $current_time;
			$week_start_date		= date_i18n( 'Y-m-d', $week_start_day );
			$week_start_time		= date_i18n( 'H:i:s', $recur_start_time );

			// Get end date time
			$get_eday_name			= $week_days[ $recur_week_end ];
			$next_end_day			= strtotime( 'next '.$get_eday_name );
			$week_end_day			= ( $curr_day_of_week != $get_eday_name ) ? $next_end_day : $current_time;
			$week_end_date			= date_i18n( 'Y-m-d', $week_end_day );
			$week_end_time			= date_i18n( 'H:i:s', $recur_end_time );
			
			// If `Week Start` is equal to `Week End` and `Current Day` is equal to `Week Start`
			if( $recur_week_start == $recur_week_end && $recur_week_start == $curr_day_number ) {

				if( $recur_start_time >= $current_time ) {

					$week_start_date	= date_i18n( 'Y-m-d', $last_start_day );
					$week_end_date		= date_i18n( 'Y-m-d', $current_time );

				} elseif( $recur_start_time <= $current_time ) {

					$week_start_date	= date_i18n( 'Y-m-d', $current_time );
					$week_end_date		= date_i18n( 'Y-m-d', $next_end_day );
				}
			}

			$results['start_time']	= $week_start_date.' '.$week_start_time;
			$results['end_time']	= $week_end_date.' '.$week_end_time;

		} else {

			$expiry_date			= date_i18n( 'Y-m-d H:i:s', strtotime(' -1 day') );
			
			$results['start_time']	= $expiry_date;
			$results['end_time']	= $expiry_date;
		}

	} else if( $recurring_type == 'custom' ) { // If `Recurring Type` is `Custom`

		// Taking some variable
		$recur_days			= ! empty( $paoc_advance_sett['recur_days'] ) ? $paoc_advance_sett['recur_days'] : array();
		$curr_day_of_week	= strtolower( current_time( 'l' ) );
		$curr_day_number	= date("w", strtotime( $curr_day_of_week ));

		// If check current day is there in `recur_days` array
		if( in_array( $curr_day_number, $recur_days ) ) {

			$results['start_time']	= date_i18n( 'Y-m-d H:i:s', $recur_start_time );
			$results['end_time']	= date_i18n( 'Y-m-d H:i:s', $recur_end_time );

		} else { // If timer day is not there then expire it

			$expiry_date			= date_i18n( 'Y-m-d H:i:s', strtotime(' -1 day') );
			
			$results['start_time']	= $expiry_date;
			$results['end_time']	= $expiry_date;
		}
	}

	return apply_filters( 'paoc_pro_popup_recurring_dates', $results, $popup_id, $paoc_advance_sett );
}

/**
 * Function to create popup HTML
 * 
 * @since 2.0
 */
function paoc_pro_generate_popup_style( $popup_id = 0 ) {

	global $paoc_design_sett, $paoc_behaviour_sett, $paoc_advance_sett, $paoc_custom_css;

	// If valid post is there
	if( empty( $popup_id ) ) {
		return false;
	}

	// Taking some data
	$style		= array(
					'bg_img'	=> '',
					'popup_img'	=> '',
					'bg_clr'	=> '',
					'clr'		=> '',
					'inline'	=> '',
				);
	$prefix			= PAOC_PRO_META_PREFIX;
	$design			= empty( $paoc_design_sett )	? get_post_meta( $popup_id, $prefix.'design', true )		: $paoc_design_sett;
	$behaviour		= empty( $paoc_behaviour_sett )	? get_post_meta( $popup_id, $prefix.'behaviour', true )		: $paoc_behaviour_sett;
	$advance		= empty( $paoc_advance_sett )	? get_post_meta( $popup_id, $prefix.'advance', true )		: $paoc_advance_sett;
	$custom_css		= empty( $paoc_custom_css )		? get_post_meta( $popup_id, $prefix.'custom_css', true )	: $paoc_custom_css;

	$hide_overlay			= ! empty( $behaviour['hide_overlay'] )		? 1 : 0;
	$popup_width			= isset( $design['width'] ) 				? $design['width']					: '';
	$popup_height			= isset( $design['height'] ) 				? $design['height']					: '';
	$fullscreen_popup		= isset( $design['fullscreen_popup'] ) 		? $design['fullscreen_popup']		: '';
	$overlay_img			=  isset( $design['overlay_img'] )			? $design['overlay_img']			: '';
	$overlay_color			=  isset( $design['overlay_color'] )		? $design['overlay_color']			: '';
	$popup_img				=  isset( $design['popup_img'] )			? $design['popup_img']				: '';
	$bg_color				=  isset( $design['bg_color'] )				? $design['bg_color']				: '';
	$mheading_fontsize		=  isset( $design['mheading_fontsize'] )	? $design['mheading_fontsize']		: '';
	$mheading_txtcolor		=  isset( $design['mheading_txtcolor'] )	? $design['mheading_txtcolor']		: '';
	$sheading_fontsize		=  isset( $design['sheading_fontsize'] )	? $design['sheading_fontsize']		: '';
	$sheading_txtcolor		=  isset( $design['sheading_txtcolor'] )	? $design['sheading_txtcolor']		: '';
	$content_color			=  isset( $design['content_color'] )		? $design['content_color']			: '';
	$secondary_cnt_clr		=  isset( $design['secondary_cnt_clr'] )	? $design['secondary_cnt_clr']		: '';
	$custom_close_txtclr	=  isset( $design['custom_close_txtclr'] )	? $design['custom_close_txtclr']	: '';
	$snote_txtcolor			=  isset( $design['snote_txtcolor'] )		? $design['snote_txtcolor']			: '';

	// Popup Goal : Collect Lead
	$form_field_lbl_clr		= isset( $design['email_lists']['field_lbl_clr'] )	? $design['email_lists']['field_lbl_clr']	: '';
	$form_field_bg_clr		= isset( $design['email_lists']['field_bg_clr'] )	? $design['email_lists']['field_bg_clr']	: '';
	$form_field_txt_clr		= isset( $design['email_lists']['field_txt_clr'] )	? $design['email_lists']['field_txt_clr']	: '';
	$form_btn_bg_clr		= isset( $design['email_lists']['btn_bg_clr'] )		? $design['email_lists']['btn_bg_clr']		: '';
	$form_btn_hbg_clr		= isset( $design['email_lists']['btn_hbg_clr'] )	? $design['email_lists']['btn_hbg_clr']		: '';
	$form_btn_txt_clr		= isset( $design['email_lists']['btn_txt_clr'] )	? $design['email_lists']['btn_txt_clr']		: '';
	$form_btn_htxt_clr		= isset( $design['email_lists']['btn_htxt_clr'] )	? $design['email_lists']['btn_htxt_clr']	: '';

	// Popup Goal : Target URL
	$url_btn1_bg_clr		=  isset( $design['target_url']['btn1_bg_clr'] )	? $design['target_url']['btn1_bg_clr']		: '';
	$url_btn1_hbg_clr		=  isset( $design['target_url']['btn1_hbg_clr'] )	? $design['target_url']['btn1_hbg_clr']		: '';
	$url_btn1_txt_clr		=  isset( $design['target_url']['btn1_txt_clr'] )	? $design['target_url']['btn1_txt_clr']		: '';
	$url_btn1_htxt_clr		=  isset( $design['target_url']['btn1_htxt_clr'] )	? $design['target_url']['btn1_htxt_clr']	: '';

	$url_btn2_bg_clr		=  isset( $design['target_url']['btn2_bg_clr'] )	? $design['target_url']['btn2_bg_clr']		: '';
	$url_btn2_hbg_clr		=  isset( $design['target_url']['btn2_hbg_clr'] )	? $design['target_url']['btn2_hbg_clr']		: '';
	$url_btn2_txt_clr		=  isset( $design['target_url']['btn2_txt_clr'] )	? $design['target_url']['btn2_txt_clr']		: '';
	$url_btn2_htxt_clr		=  isset( $design['target_url']['btn2_htxt_clr'] )	? $design['target_url']['btn2_htxt_clr']	: '';

	// Popup Goal : Phone Call
	$call_btn_bg_clr		=  isset( $design['phone_calls']['btn_bg_clr'] )	? $design['phone_calls']['btn_bg_clr']		: '';
	$call_btn_hbg_clr		=  isset( $design['phone_calls']['btn_hbg_clr'] )	? $design['phone_calls']['btn_hbg_clr']		: '';
	$call_btn_txt_clr		=  isset( $design['phone_calls']['btn_txt_clr'] )	? $design['phone_calls']['btn_txt_clr']		: '';
	$call_btn_htxt_clr		=  isset( $design['phone_calls']['btn_htxt_clr'] )	? $design['phone_calls']['btn_htxt_clr']	: '';

	// Show Credit
	$show_credit = ! empty( $advance['show_credit'] ) ? 1 : 0;

	// Custom CSS
	$custom_css = isset( $custom_css ) ? $custom_css : '';

	// Create Popup Background Style
	if( $popup_img ) {
		$style['bg_img']	= "background-image: url({$popup_img}); ";
		$style['popup_img']	= $popup_img;
	}
	if( $bg_color ) {
		$style['bg_clr']	= "background-color:{$bg_color}; ";
		$style['clr']		= "color:{$bg_color}; ";
	}

	if( $popup_width ) {
		if( ! $hide_overlay ) {
			$style['inline'] .= ".paoc-popup-{$popup_id}{max-width: {$popup_width};}";
		} else {
			$style['inline'] .= ".paoc-cb-popup-{$popup_id}.paoc-hide-overlay.custombox-content {max-width: {$popup_width};}";
		}
	}

	if( $popup_height && ! $fullscreen_popup ) {

		$style['inline'] .= ".paoc-popup-{$popup_id} {height: {$popup_height}px;}";
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-inr-wrap{height: 100%;}";
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-inr{overflow-y:auto;}";
	}

	// Show Credit
	if( $show_credit ) {
		if( ! $hide_overlay ) {
			$style['inline'] .= ".custombox-y-bottom .paoc-popup-{$popup_id}{margin-bottom: 34px;}";
			$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-inline-popup{margin-bottom: 49px;}";
		}
	}

	if ( $hide_overlay ) {
		if( $show_credit ) {
			$style['inline'] .= ".admin-bar .paoc-popup-{$popup_id}{max-height: calc(100vh - 66px) !important; margin-bottom: 34px;}";
			$style['inline'] .= ".paoc-popup-{$popup_id}{max-height: calc(100vh - 32px) !important; margin-bottom: 34px !important;}";
			$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-inline-popup{margin-bottom: 49px !important;}";
		} else {
			$style['inline'] .= ".admin-bar .paoc-popup-{$popup_id}{max-height: calc(100vh - 32px) !important;}";
		}
	}

	// Overlay Style
	if( $overlay_img ) {
		$style['inline'] .= ".paoc-popup-overlay-{$popup_id} {background-image: url({$overlay_img}); background-repeat:repeat; background-position: center center;}";
	}
	if( $overlay_color ) {
		$style['inline'] .= ".paoc-popup-overlay-{$popup_id} {background-color:{$overlay_color} !important;}";
	}

	// Main Heading Style
	if( $mheading_fontsize ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-mheading {font-size: {$mheading_fontsize}px;}";
	}
	if( $mheading_txtcolor ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-mheading {color: {$mheading_txtcolor};}";
	}

	// Sub Heading Style
	if( $sheading_fontsize ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-sheading {font-size: {$sheading_fontsize}px;}";
	}
	if( $sheading_txtcolor ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-sheading {color: {$sheading_txtcolor};}";
	}

	// Content Style
	if( $content_color ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-content, .paoc-popup-{$popup_id} .paoc-popup-content p {color: {$content_color}; }";
	}

	// Secondary Content Styel
	if( $secondary_cnt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-secondary-con, .paoc-popup-{$popup_id} .paoc-secondary-con p {color: {$secondary_cnt_clr}; }";
	}

	// Custom Close Style
	if( $custom_close_txtclr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-cus-close-txt {color: {$custom_close_txtclr}; }";
	}

	// Security Note Style
	if( $snote_txtcolor ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-snote {color: {$snote_txtcolor}; }";
	}

	// Popup Goal : Collect Lead
	if( $form_field_lbl_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-popup-inr .paoc-form-lbl, .paoc-popup-{$popup_id} .paoc-form-ckbox-lbl { color: {$form_field_lbl_clr}; }";
	}
	if( $form_field_bg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-inp { background-color: {$form_field_bg_clr}; }";
	}
	if( $form_field_txt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-inp { color: {$form_field_txt_clr}; }";
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-inp::placeholder, .paoc-popup-{$popup_id} .paoc-form-inp::-webkit-input-placeholder { color: {$form_field_txt_clr}; }";
	}

	if( $form_btn_bg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit { background-color: {$form_btn_bg_clr}; } ";
	}
	if( $form_btn_txt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit { color: {$form_btn_txt_clr}; } ";
	}

	if( $form_btn_hbg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit:hover, .paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit:focus, .paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit:active { background-color: {$form_btn_hbg_clr}; }";
	}
	if( $form_btn_htxt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit:hover, .paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit:focus, .paoc-popup-{$popup_id} .paoc-form-fields .paoc-form-submit:active { color: {$form_btn_htxt_clr}; }";
	}

	// Popup Goal : Target URL
	if( $url_btn1_bg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn { background-color: {$url_btn1_bg_clr}; }";
	}
	if( $url_btn1_txt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn { color: {$url_btn1_txt_clr}; }";
	}

	if( $url_btn1_hbg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn:hover, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn:focus, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn:active { background-color: {$url_btn1_hbg_clr}; }";
	}
	if( $url_btn1_htxt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn:hover, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn:focus, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-1 .paoc-btn:active { color: {$url_btn1_htxt_clr}; }";
	}

	if( $url_btn2_bg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn { background-color: {$url_btn2_bg_clr}; }";
	}
	if( $url_btn2_txt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn { color: {$url_btn2_txt_clr}; }";
	}

	if( $url_btn2_hbg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn:hover, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn:focus, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn:active { background-color: {$url_btn2_hbg_clr}; }";
	}
	if( $url_btn2_htxt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn:hover, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn:focus, .paoc-popup-{$popup_id}.paoc-popup-target-url .paoc-btn-2 .paoc-btn:active { color: {$url_btn2_htxt_clr}; }";
	}

	// Popup Goal : Phone Call
	if( $call_btn_bg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1 { background-color: {$call_btn_bg_clr}; }";
	}
	if( $call_btn_txt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1 { color: {$call_btn_txt_clr}; }";
	}

	if( $call_btn_hbg_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1:hover, .paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1:focus, .paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1:active { background-color: {$call_btn_hbg_clr}; }";
	}
	if( $call_btn_htxt_clr ) {
		$style['inline'] .= ".paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1:hover, .paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1:focus, .paoc-popup-{$popup_id}.paoc-popup-phone-calls .paoc-btn-1:active { color: {$call_btn_htxt_clr}; }";
	}

	// HTML Popup CSS
	$style['inline'] .= ".paoc-pro-popup-cust-{$popup_id} {cursor: pointer;}";

	// Custom CSS
	$style['inline'] .= $custom_css;

	return $style;
}