/**
 * Enhanced Prop Bet Generator JavaScript
 * Version: 2.0.0
 */

(function($) {
    'use strict';
    
    // Ensure pbg_ajax is available
    if (typeof pbg_ajax === 'undefined') {
        console.error('PBG: Ajax configuration not found');
        return;
    }
    
    /**
     * Main Plugin Class
     */
    class PropBetGenerator {
        constructor(container) {
            this.container = $(container);
            this.containerId = container.id;
            this.data = {
                sports: [],
                games: [],
                players: [],
                propTypes: [],
                selectedSport: '',
                selectedGame: '',
                selectedPlayer: '',
                selectionMode: 'game'
            };
            
            this.init();
        }
        
        init() {
            this.bindEvents();
            this.loadSports();
            this.checkApiStatus();
        }
        
        bindEvents() {
            const self = this;
            
            // Form submission
            this.container.find('#pbg-form').on('submit', function(e) {
                e.preventDefault();
                self.handleSubmit();
            });
            
            // Sport selection
            this.container.find('#pbg-sport').on('change', function() {
                self.handleSportChange($(this).val());
            });
            
            // Game selection
            this.container.find('#pbg-game').on('change', function() {
                self.handleGameChange($(this).val());
            });
            
            // Selection mode change
            this.container.find('input[name="selection_mode"]').on('change', function() {
                self.handleModeChange($(this).val());
            });
            
            // Manual player input
            this.container.find('#pbg-player-manual').on('input', function() {
                self.updateSubmitButton();
            });
            
            // Prop type checkboxes
            this.container.on('change', 'input[name="prop_types[]"]', function() {
                self.updateSubmitButton();
            });
        }
        
        async loadSports() {
            try {
                const response = await this.ajaxRequest('pbg_get_sports', {});
                if (response.success) {
                    this.data.sports = response.data;
                    this.populateSports();
                }
            } catch (error) {
                this.showError('Failed to load sports');
                console.error('Error loading sports:', error);
            }
        }
        
        populateSports() {
            const select = this.container.find('#pbg-sport');
            select.html('<option value="">Choose a sport...</option>');
            
            this.data.sports.forEach(sport => {
                const option = $('<option></option>')
                    .attr('value', sport.key)
                    .text(this.getSportDisplayName(sport.key));
                select.append(option);
            });
        }
        
        async handleSportChange(sport) {
            this.data.selectedSport = sport;
            
            if (sport) {
                this.container.find('.pbg-selection-mode').show();
                await this.loadGames();
                this.loadPropTypes();
            } else {
                this.hideGameElements();
            }
        }
        
        async loadGames() {
            try {
                const response = await this.ajaxRequest('pbg_get_games', {
                    sport: this.data.selectedSport
                });
                
                if (response.success) {
                    this.data.games = response.data;
                    this.populateGames();
                }
            } catch (error) {
                this.showError('Failed to load games');
                console.error('Error loading games:', error);
            }
        }
        
        populateGames() {
            const select = this.container.find('#pbg-game');
            select.html('<option value="">Choose a game...</option>');
            
            this.data.games.forEach((game, index) => {
                let text = `${game.away_team} @ ${game.home_team}`;
                if (game.commence_time) {
                    const date = new Date(game.commence_time);
                    text += ` - ${date.toLocaleDateString()}`;
                }
                
                const option = $('<option></option>')
                    .attr('value', index)
                    .text(text);
                select.append(option);
            });
            
            this.container.find('.pbg-game-selection').show();
        }
        
        handleGameChange(gameIndex) {
            const index = parseInt(gameIndex);
            if (!isNaN(index) && this.data.games[index]) {
                this.data.selectedGame = index;
                this.data.players = this.data.games[index].all_players || [];
                this.populatePlayers();
            }
        }
        
        populatePlayers() {
            const container = this.container.find('#pbg-player-list');
            
            if (this.data.players.length === 0) {
                container.html('<p class="pbg-loading">No players available for this game.</p>');
                return;
            }
            
            container.empty();
            
            this.data.players.forEach(player => {
                const button = $(`
                    <button type="button" class="pbg-player-btn">
                        <span class="pbg-player-name">${player.name}</span>
                        <div class="pbg-player-info">
                            <span class="pbg-player-position">${player.position}</span>
                            <span class="pbg-player-team">${player.team}</span>
                        </div>
                    </button>
                `);
                
                button.on('click', () => {
                    this.selectPlayer(player.name, button);
                });
                
                container.append(button);
            });
            
            this.container.find('.pbg-player-selection').show();
        }
        
        selectPlayer(playerName, buttonElement) {
            // Remove previous selection
            this.container.find('.pbg-player-btn').removeClass('pbg-selected');
            
            // Add selection to clicked button
            buttonElement.addClass('pbg-selected');
            this.data.selectedPlayer = playerName;
            
            this.updateSubmitButton();
        }
        
        loadPropTypes() {
            const sport = this.data.sports.find(s => s.key === this.data.selectedSport);
            if (sport && sport.prop_types) {
                this.data.propTypes = sport.prop_types;
                this.populatePropTypes();
            }
        }
        
        populatePropTypes() {
            const container = this.container.find('#pbg-prop-list');
            container.empty();
            
            this.data.propTypes.forEach(prop => {
                const label = $(`
                    <label class="pbg-prop-label">
                        <input type="checkbox" name="prop_types[]" value="${prop.id}">
                        <span class="pbg-prop-name">${prop.name}</span>
                        <span class="pbg-prop-line">Typical: ${prop.typical_line}</span>
                    </label>
                `);
                container.append(label);
            });
            
            // Auto-select first 3 props
            const checkboxes = container.find('input[type="checkbox"]');
            checkboxes.slice(0, Math.min(3, checkboxes.length)).prop('checked', true);
            
            this.container.find('.pbg-prop-types').show();
            this.updateSubmitButton();
        }
        
        handleModeChange(mode) {
            this.data.selectionMode = mode;
            
            if (mode === 'game') {
                this.container.find('.pbg-game-selection').show();
                this.container.find('.pbg-player-selection').show();
                this.container.find('.pbg-manual-entry').hide();
            } else {
                this.container.find('.pbg-game-selection').hide();
                this.container.find('.pbg-player-selection').hide();
                this.container.find('.pbg-manual-entry').show();
            }
            
            this.updateSubmitButton();
        }
        
        updateSubmitButton() {
            const button = this.container.find('#pbg-submit');
            const sport = this.data.selectedSport;
            const player = this.data.selectionMode === 'game' ? 
                this.data.selectedPlayer : 
                this.container.find('#pbg-player-manual').val().trim();
            const props = this.container.find('input[name="prop_types[]"]:checked');
            
            button.prop('disabled', !sport || !player || props.length === 0);
        }
        
        async handleSubmit() {
            const button = this.container.find('#pbg-submit');
            const buttonText = button.find('.pbg-btn-text');
            const buttonLoading = button.find('.pbg-btn-loading');
            
            // Get form data
            const playerName = this.data.selectionMode === 'game' ? 
                this.data.selectedPlayer : 
                this.container.find('#pbg-player-manual').val().trim();
            
            const propTypes = this.container.find('input[name="prop_types[]"]:checked')
                .map(function() { return $(this).val(); }).get();
            
            // Show loading state
            button.prop('disabled', true);
            buttonText.hide();
            buttonLoading.show();
            
            // Clear previous results and errors
            this.container.find('#pbg-results').html(`
                <div class="pbg-loading-results">
                    <span class="pbg-spinner"></span>
                    <p>Generating AI recommendations...</p>
                </div>
            `);
            this.container.find('#pbg-error').hide();
            
            try {
                const response = await this.ajaxRequest('pbg_generate_props', {
                    sport: this.data.selectedSport,
                    player_name: playerName,
                    prop_types: propTypes,
                    game_id: this.data.selectedGame
                });
                
                if (response.success) {
                    this.displayResults(response.data);
                } else {
                    this.showError('Failed to generate recommendations: ' + (response.data || 'Unknown error'));
                }
            } catch (error) {
                this.showError('Network error occurred while generating recommendations');
                console.error('Error generating props:', error);
            } finally {
                // Reset button state
                button.prop('disabled', false);
                buttonText.show();
                buttonLoading.hide();
                this.updateSubmitButton();
            }
        }
        
        displayResults(data) {
            const container = this.container.find('#pbg-results');
            
            if (!data.recommendations || data.recommendations.length === 0) {
                container.html(`
                    <div class="pbg-results-empty">
                        <div class="pbg-empty-icon">❌</div>
                        <p>No recommendations generated. Please try again.</p>
                    </div>
                `);
                return;
            }
            
            let html = '';
            
            data.recommendations.forEach(rec => {
                const confidenceClass = `pbg-confidence-${rec.confidence.toLowerCase()}`;
                
                html += `
                    <div class="pbg-recommendation">
                        <div class="pbg-rec-header">
                            <div class="pbg-rec-player">
                                <h4>${this.escapeHtml(rec.player_name)}</h4>
                                <p>${this.escapeHtml(rec.prop_display_name)}</p>
                            </div>
                            <div class="pbg-rec-confidence ${confidenceClass}">
                                ${this.escapeHtml(rec.confidence)} Confidence
                            </div>
                        </div>
                        
                        <div class="pbg-rec-stats">
                            <div class="pbg-stat-box">
                                <label>Recommended Line</label>
                                <span class="pbg-stat-value">${rec.recommended_line}</span>
                            </div>
                            ${rec.best_odds ? `
                                <div class="pbg-stat-box">
                                    <label>Best Odds</label>
                                    <span class="pbg-stat-value pbg-odds-value">${this.formatOdds(rec.best_odds)}</span>
                                    ${rec.sportsbook ? `<small>${this.escapeHtml(rec.sportsbook)}</small>` : ''}
                                </div>
                            ` : ''}
                        </div>
                        
                        <div class="pbg-rec-explanation">
                            <label>🧠 AI Analysis</label>
                            <p>${this.escapeHtml(rec.explanation)}</p>
                        </div>
                    </div>
                `;
            });
            
            container.html(html);
        }
        
        showError(message) {
            const errorDiv = this.container.find('#pbg-error');
            errorDiv.text(message).show();
            
            // Auto-hide after 5 seconds
            setTimeout(() => {
                errorDiv.fadeOut();
            }, 5000);
        }
        
        async checkApiStatus() {
            // In a real implementation, this would make actual API calls
            // For now, we'll use a simple check
            try {
                const response = await this.ajaxRequest('pbg_get_sports', {});
                if (response.success) {
                    this.container.find('#pbg-odds-status .pbg-status-value').text('Connected');
                    this.container.find('#pbg-ai-status .pbg-status-value').text('Connected');
                } else {
                    throw new Error('API check failed');
                }
            } catch (error) {
                this.container.find('#pbg-odds-status .pbg-status-value').text('Disconnected');
                this.container.find('#pbg-ai-status .pbg-status-value').text('Disconnected');
            }
        }
        
        getSportDisplayName(sportKey) {
            const names = {
                'americanfootball_nfl': 'NFL',
                'basketball_nba': 'NBA',
                'baseball_mlb': 'MLB',
                'icehockey_nhl': 'NHL'
            };
            return names[sportKey] || sportKey;
        }
        
        hideGameElements() {
            this.container.find('.pbg-selection-mode').hide();
            this.container.find('.pbg-game-selection').hide();
            this.container.find('.pbg-player-selection').hide();
            this.container.find('.pbg-manual-entry').hide();
            this.container.find('.pbg-prop-types').hide();
            this.updateSubmitButton();
        }
        
        async ajaxRequest(action, data) {
            return new Promise((resolve, reject) => {
                const requestData = {
                    action: action,
                    nonce: pbg_ajax.nonce,
                    ...data
                };
                
                $.ajax({
                    url: pbg_ajax.ajax_url,
                    method: 'POST',
                    data: requestData,
                    dataType: 'json',
                    timeout: 30000,
                    success: function(response) {
                        resolve(response);
                    },
                    error: function(xhr, status, error) {
                        console.error('AJAX Error:', status, error);
                        reject(new Error(`AJAX request failed: ${status}`));
                    }
                });
            });
        }
        
        escapeHtml(text) {
            if (typeof text !== 'string') return text;
            const div = document.createElement('div');
            div.textContent = text;
            return div.innerHTML;
        },
        
        formatOdds(odds) {
            // If odds is already in American format (starts with + or -), return as is
            if (typeof odds === 'string' && (odds.startsWith('+') || odds.startsWith('-'))) {
                return odds;
            }
            
            // If it's a decimal number, convert to American format
            const decimal = parseFloat(odds);
            if (isNaN(decimal)) return odds;
            
            if (decimal >= 2.0) {
                return '+' + Math.round((decimal - 1) * 100);
            } else {
                return Math.round(-100 / (decimal - 1));
            }
        }
    }
    
    /**
     * Initialize plugin instances
     */
    $(document).ready(function() {
        $('.prop-bet-generator').each(function() {
            if (this.id) {
                new PropBetGenerator(this);
            }
        });
    });
    
    // Export for global access if needed
    window.PropBetGenerator = PropBetGenerator;
    
})(jQuery);
