<?php
/**
 * AI Service Class for generating prop bet recommendations
 */

if (!defined('ABSPATH')) {
    exit;
}

class PBG_AI_Service {
    
    private $openai_api_key;
    private $api_url = 'https://api.openai.com/v1/chat/completions';
    
    public function __construct() {
        $this->openai_api_key = get_option('pbg_openai_api_key', '');
    }
    
    public function generate_recommendations($player_stats, $sport, $prop_types) {
        if (empty($this->openai_api_key)) {
            // Use fallback if no API key
            return $this->generate_fallback_recommendations($player_stats, $sport, $prop_types);
        }
        
        try {
            $ai_recommendations = $this->call_openai_api($player_stats, $sport, $prop_types);
            if ($ai_recommendations && is_array($ai_recommendations)) {
                return $ai_recommendations;
            }
        } catch (Exception $e) {
            error_log('PBG AI Service Error: ' . $e->getMessage());
            // Re-throw the exception for better debugging
            throw $e;
        }
        
        // Fallback to manual recommendations
        return $this->generate_fallback_recommendations($player_stats, $sport, $prop_types);
    }
    
    private function call_openai_api($player_stats, $sport, $prop_types) {
        $api_service = new PBG_API_Service();
        $available_props = $api_service->get_prop_types($sport);
        
        $system_message = "You are an expert sports betting analyst specializing in {$sport} prop bets. Analyze the provided player statistics and generate intelligent prop bet recommendations. Consider season averages, recent performance patterns, position-specific expectations, and league averages. For each recommendation, provide: player name, specific prop type, recommended line with decimal precision, confidence level (Low/Medium/High) based on data strength, detailed explanation with specific statistical reasoning, and realistic betting odds. Be data-driven and conservative with High confidence ratings.";
        
        $prop_types_context = "";
        foreach ($available_props as $prop) {
            $prop_types_context .= "- {$prop['id']}: {$prop['name']} (typical line: {$prop['typical_line']})\n";
        }
        
        $user_message = "Player: {$player_stats['player_name']}\n";
        $user_message .= "Position: {$player_stats['position']}\n";
        $user_message .= "Team: {$player_stats['team']}\n";
        $user_message .= "Sport: {$sport}\n\n";
        $user_message .= "Season Statistics:\n" . json_encode($player_stats['season_stats'], JSON_PRETTY_PRINT) . "\n\n";
        $user_message .= "Available Prop Types for {$sport}:\n{$prop_types_context}\n";
        $user_message .= "Requested Prop Types: " . implode(', ', $prop_types) . "\n\n";
        $user_message .= "Please analyze this data and provide 2-3 prop bet recommendations from the requested types in JSON format:\n";
        $user_message .= '{"recommendations": [{"player_name": "' . $player_stats['player_name'] . '", "prop_type": "specific_prop_id", "prop_display_name": "Human Readable Prop Name", "recommended_line": 25.5, "confidence": "Medium", "explanation": "Detailed statistical analysis with specific numbers...", "best_odds": "1.85", "sportsbook": "DraftKings"}]}';
        $user_message .= "\n\nIMPORTANT: For best_odds, provide decimal odds (like 1.85, 2.10, etc.) - they will be converted to American format automatically.";
        
        $data = array(
            'model' => 'gpt-4o-mini',
            'messages' => array(
                array('role' => 'system', 'content' => $system_message),
                array('role' => 'user', 'content' => $user_message)
            ),
            'max_tokens' => 1000,
            'temperature' => 0.7
        );
        
        $args = array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $this->openai_api_key
            ),
            'body' => json_encode($data),
            'timeout' => 30
        );
        
        $response = wp_remote_post($this->api_url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('OpenAI API request failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            $body = wp_remote_retrieve_body($response);
            throw new Exception("OpenAI API returned error {$response_code}: {$body}");
        }
        
        $body = wp_remote_retrieve_body($response);
        $result = json_decode($body, true);
        
        if (!isset($result['choices'][0]['message']['content'])) {
            throw new Exception('Invalid OpenAI API response: ' . $body);
        }
        
        $content = $result['choices'][0]['message']['content'];
        
        // Extract JSON from response
        if (strpos($content, '```json') !== false) {
            preg_match('/```json\s*(.*?)\s*```/s', $content, $matches);
            $content = isset($matches[1]) ? $matches[1] : $content;
        } elseif (strpos($content, '```') !== false) {
            preg_match('/```\s*(.*?)\s*```/s', $content, $matches);
            $content = isset($matches[1]) ? $matches[1] : $content;
        }
        
        $ai_data = json_decode($content, true);
        
        if (!isset($ai_data['recommendations'])) {
            throw new Exception('No recommendations found in AI response');
        }
        
        $recommendations = array();
        foreach ($ai_data['recommendations'] as $rec) {
            $prop_info = null;
            foreach ($available_props as $prop) {
                if ($prop['id'] === $rec['prop_type']) {
                    $prop_info = $prop;
                    break;
                }
            }
            
            // Convert decimal odds to American format if provided
            $best_odds = null;
            if (isset($rec['best_odds'])) {
                $decimal_odds = floatval($rec['best_odds']);
                $best_odds = $this->convert_decimal_to_american($decimal_odds);
            }
            
            $recommendations[] = array(
                'player_name' => $rec['player_name'],
                'prop_type' => $rec['prop_type'],
                'prop_display_name' => isset($rec['prop_display_name']) ? $rec['prop_display_name'] : ($prop_info ? $prop_info['name'] : $rec['prop_type']),
                'recommended_line' => floatval($rec['recommended_line']),
                'confidence' => $rec['confidence'],
                'explanation' => $rec['explanation'],
                'best_odds' => $best_odds,
                'sportsbook' => isset($rec['sportsbook']) ? $rec['sportsbook'] : null,
                'player_stats' => $player_stats['season_stats']
            );
        }
        
        return $recommendations;
    }
    
    private function generate_fallback_recommendations($player_stats, $sport, $prop_types) {
        $api_service = new PBG_API_Service();
        $available_props = $api_service->get_prop_types($sport);
        
        // Filter to requested props or use first 3 available
        $target_props = array();
        foreach ($available_props as $prop) {
            if (in_array($prop['id'], $prop_types)) {
                $target_props[] = $prop;
            }
        }
        
        if (empty($target_props)) {
            $target_props = array_slice($available_props, 0, 3);
        }
        
        $recommendations = array();
        
        foreach (array_slice($target_props, 0, 3) as $prop_info) {
            $prop_id = $prop_info['id'];
            $prop_name = $prop_info['name'];
            $typical_line = $prop_info['typical_line'];
            
            // Calculate recommendation based on actual stats
            $stats = $player_stats['season_stats'];
            $recommended_line = $typical_line;
            $confidence = 'Medium';
            $explanation = "Based on season averages and league trends for " . strtolower($prop_name) . ".";
            
            // Sport-specific logic
            if ($sport === 'americanfootball_nfl') {
                if ($prop_id === 'passing_yards' && isset($stats['passing_yards'])) {
                    $avg_yards = $stats['passing_yards'] / 16; // Assume 16 games
                    $recommended_line = $avg_yards * 0.95;
                    $confidence = $avg_yards > 280 ? 'High' : 'Medium';
                    $explanation = "{$player_stats['player_name']} averages " . number_format($avg_yards, 1) . " passing yards per game this season.";
                } elseif ($prop_id === 'rushing_yards' && isset($stats['rushing_yards'])) {
                    $avg_yards = $stats['rushing_yards'] / 16;
                    $recommended_line = $avg_yards * 0.92;
                    $confidence = $avg_yards > 90 ? 'High' : 'Medium';
                    $explanation = "Player averages " . number_format($avg_yards, 1) . " rushing yards per game, slight under recommended.";
                }
            } elseif ($sport === 'basketball_nba') {
                if ($prop_id === 'points' && isset($stats['points_per_game'])) {
                    $avg_points = $stats['points_per_game'];
                    $recommended_line = $avg_points - 0.5;
                    $confidence = $avg_points > 25 ? 'High' : 'Medium';
                    $explanation = "{$player_stats['player_name']} averages " . number_format($avg_points, 1) . " points per game this season.";
                } elseif ($prop_id === 'rebounds' && isset($stats['rebounds_per_game'])) {
                    $avg_rebounds = $stats['rebounds_per_game'];
                    $recommended_line = $avg_rebounds + 0.5;
                    $confidence = 'Medium';
                    $explanation = "Player averages " . number_format($avg_rebounds, 1) . " rebounds, favorable matchup suggests over.";
                }
            }
            
            // Generate decimal odds and convert to American
            $decimal_odds = 1.85 + (crc32($prop_id) % 20) / 100;
            $american_odds = $this->convert_decimal_to_american($decimal_odds);
            
            $recommendations[] = array(
                'player_name' => $player_stats['player_name'],
                'prop_type' => $prop_id,
                'prop_display_name' => $prop_name,
                'recommended_line' => round($recommended_line, 1),
                'confidence' => $confidence,
                'explanation' => $explanation,
                'best_odds' => $american_odds,
                'sportsbook' => array('DraftKings', 'FanDuel', 'BetMGM', 'Caesars')[crc32($prop_id) % 4],
                'player_stats' => $stats
            );
        }
        
        return $recommendations;
    }
    
    /**
     * Convert decimal odds to American odds format
     */
    private function convert_decimal_to_american($decimal_odds) {
        if ($decimal_odds >= 2.0) {
            // Positive American odds
            return '+' . round(($decimal_odds - 1) * 100);
        } else {
            // Negative American odds
            return round(-100 / ($decimal_odds - 1));
        }
    }
}
