<?php
/**
 * API Service Class for handling external API calls
 */

if (!defined('ABSPATH')) {
    exit;
}

class PBG_API_Service {
    
    private $odds_api_key;
    private $base_url;
    
    public function __construct() {
        $this->odds_api_key = get_option('pbg_odds_api_key', '');
        $this->base_url = 'https://api.the-odds-api.com/v4';
    }
    
    public function get_sports() {
        if (empty($this->odds_api_key)) {
            throw new Exception('TheOddsAPI key is not configured');
        }
        
        $cache_key = 'pbg_sports';
        $cached = $this->get_cached_data($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $url = $this->base_url . '/sports';
        $args = array(
            'headers' => array(),
            'timeout' => 30
        );
        
        $url = add_query_arg(array('apiKey' => $this->odds_api_key), $url);
        
        // Log the request URL for debugging (remove in production)
        error_log('PBG: Making request to ' . $url);
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            throw new Exception('TheOddsAPI request failed: ' . $response->get_error_message());
        }
        
        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            $body = wp_remote_retrieve_body($response);
            throw new Exception("TheOddsAPI returned error {$response_code}: {$body}");
        }
        
        $body = wp_remote_retrieve_body($response);
        
        // Log the raw response for debugging
        error_log('PBG: Raw API response: ' . substr($body, 0, 500));
        
        $sports = json_decode($body, true);
        
        if (!is_array($sports)) {
            throw new Exception('Invalid response from TheOddsAPI: ' . $body);
        }
        
        // Log how many sports were returned
        error_log('PBG: Total sports returned: ' . count($sports));
        
        // Filter for major sports and add prop types
        $major_sports = array('americanfootball_nfl', 'basketball_nba', 'baseball_mlb', 'icehockey_nhl');
        $filtered_sports = array();
        
        foreach ($sports as $sport) {
            error_log('PBG: Found sport: ' . $sport['key']);
            if (in_array($sport['key'], $major_sports)) {
                $sport['prop_types'] = $this->get_prop_types($sport['key']);
                $filtered_sports[] = $sport;
                error_log('PBG: Added major sport: ' . $sport['key']);
            }
        }
        
        error_log('PBG: Filtered sports count: ' . count($filtered_sports));
        
        // If no major sports found, let's be more flexible and include American sports
        if (empty($filtered_sports)) {
            foreach ($sports as $sport) {
                // Look for any American football, basketball, baseball, or hockey
                if (strpos($sport['key'], 'football') !== false || 
                    strpos($sport['key'], 'basketball') !== false || 
                    strpos($sport['key'], 'baseball') !== false || 
                    strpos($sport['key'], 'hockey') !== false) {
                    
                    // Map to our supported prop types
                    $sport_key = $sport['key'];
                    if (strpos($sport_key, 'football') !== false) {
                        $sport['prop_types'] = $this->get_prop_types('americanfootball_nfl');
                    } elseif (strpos($sport_key, 'basketball') !== false) {
                        $sport['prop_types'] = $this->get_prop_types('basketball_nba');
                    } elseif (strpos($sport_key, 'baseball') !== false) {
                        $sport['prop_types'] = $this->get_prop_types('baseball_mlb');
                    } elseif (strpos($sport_key, 'hockey') !== false) {
                        $sport['prop_types'] = $this->get_prop_types('icehockey_nhl');
                    }
                    
                    $filtered_sports[] = $sport;
                    error_log('PBG: Added flexible sport: ' . $sport['key']);
                }
            }
        }
        
        if (empty($filtered_sports)) {
            throw new Exception('No major sports found. Total sports returned: ' . count($sports) . '. Available sports: ' . implode(', ', array_column($sports, 'key')));
        }
        
        $this->cache_data($cache_key, $filtered_sports, 3600); // Cache for 1 hour
        
        return $filtered_sports;
    }
    
    public function get_games($sport) {
        $cache_key = 'pbg_games_' . $sport;
        $cached = $this->get_cached_data($cache_key);
        
        if ($cached !== false) {
            return $cached;
        }
        
        $url = $this->base_url . '/sports/' . $sport . '/odds';
        $args = array(
            'headers' => array(),
            'timeout' => 30
        );
        
        $query_args = array(
            'apiKey' => $this->odds_api_key,
            'regions' => 'us',
            'markets' => 'h2h',
            'oddsFormat' => 'decimal'
        );
        
        $url = add_query_arg($query_args, $url);
        
        $response = wp_remote_get($url, $args);
        
        if (is_wp_error($response)) {
            return array();
        }
        
        $body = wp_remote_retrieve_body($response);
        $games = json_decode($body, true);
        
        if (!is_array($games)) {
            return array();
        }
        
        $this->cache_data($cache_key, $games, 900); // Cache for 15 minutes
        
        return $games;
    }
    
    public function get_enhanced_games($sport) {
        $games = $this->get_games($sport);
        $player_service = new PBG_Player_Service();
        
        $enhanced_games = array();
        
        foreach ($games as $game) {
            $home_team = $game['home_team'];
            $away_team = $game['away_team'];
            
            $home_players = $player_service->get_players_by_team($sport, $home_team);
            $away_players = $player_service->get_players_by_team($sport, $away_team);
            
            $game['home_players'] = $home_players;
            $game['away_players'] = $away_players;
            $game['all_players'] = array_merge($home_players, $away_players);
            
            $enhanced_games[] = $game;
        }
        
        return $enhanced_games;
    }
    
    public function get_prop_types($sport) {
        $prop_types = array(
            'americanfootball_nfl' => array(
                array('id' => 'passing_yards', 'name' => 'Passing Yards', 'typical_line' => 250.5),
                array('id' => 'passing_touchdowns', 'name' => 'Passing Touchdowns', 'typical_line' => 1.5),
                array('id' => 'rushing_yards', 'name' => 'Rushing Yards', 'typical_line' => 75.5),
                array('id' => 'rushing_touchdowns', 'name' => 'Rushing Touchdowns', 'typical_line' => 0.5),
                array('id' => 'receiving_yards', 'name' => 'Receiving Yards', 'typical_line' => 65.5),
                array('id' => 'receptions', 'name' => 'Receptions', 'typical_line' => 5.5),
                array('id' => 'receiving_touchdowns', 'name' => 'Receiving Touchdowns', 'typical_line' => 0.5),
                array('id' => 'interceptions', 'name' => 'Interceptions Thrown', 'typical_line' => 0.5),
                array('id' => 'sacks', 'name' => 'Sacks (Defense)', 'typical_line' => 0.5),
                array('id' => 'field_goals', 'name' => 'Field Goals Made', 'typical_line' => 1.5)
            ),
            'basketball_nba' => array(
                array('id' => 'points', 'name' => 'Points', 'typical_line' => 25.5),
                array('id' => 'rebounds', 'name' => 'Total Rebounds', 'typical_line' => 8.5),
                array('id' => 'assists', 'name' => 'Assists', 'typical_line' => 6.5),
                array('id' => 'three_pointers', 'name' => 'Three-Pointers Made', 'typical_line' => 2.5),
                array('id' => 'steals', 'name' => 'Steals', 'typical_line' => 1.5),
                array('id' => 'blocks', 'name' => 'Blocks', 'typical_line' => 0.5),
                array('id' => 'turnovers', 'name' => 'Turnovers', 'typical_line' => 3.5),
                array('id' => 'free_throws', 'name' => 'Free Throws Made', 'typical_line' => 4.5),
                array('id' => 'double_double', 'name' => 'Double-Double', 'typical_line' => 0.5),
                array('id' => 'triple_double', 'name' => 'Triple-Double', 'typical_line' => 0.5)
            ),
            'baseball_mlb' => array(
                array('id' => 'hits', 'name' => 'Hits', 'typical_line' => 1.5),
                array('id' => 'home_runs', 'name' => 'Home Runs', 'typical_line' => 0.5),
                array('id' => 'rbis', 'name' => 'RBIs', 'typical_line' => 1.5),
                array('id' => 'runs', 'name' => 'Runs Scored', 'typical_line' => 1.5),
                array('id' => 'stolen_bases', 'name' => 'Stolen Bases', 'typical_line' => 0.5),
                array('id' => 'strikeouts_batter', 'name' => 'Strikeouts (Batter)', 'typical_line' => 1.5),
                array('id' => 'strikeouts_pitcher', 'name' => 'Strikeouts (Pitcher)', 'typical_line' => 6.5),
                array('id' => 'earned_runs', 'name' => 'Earned Runs Allowed', 'typical_line' => 2.5),
                array('id' => 'walks_pitcher', 'name' => 'Walks Allowed', 'typical_line' => 2.5),
                array('id' => 'innings_pitched', 'name' => 'Innings Pitched', 'typical_line' => 5.5)
            ),
            'icehockey_nhl' => array(
                array('id' => 'goals', 'name' => 'Goals', 'typical_line' => 0.5),
                array('id' => 'assists', 'name' => 'Assists', 'typical_line' => 0.5),
                array('id' => 'points', 'name' => 'Points (Goals + Assists)', 'typical_line' => 1.5),
                array('id' => 'shots_on_goal', 'name' => 'Shots on Goal', 'typical_line' => 3.5),
                array('id' => 'saves', 'name' => 'Saves (Goalie)', 'typical_line' => 25.5),
                array('id' => 'penalty_minutes', 'name' => 'Penalty Minutes', 'typical_line' => 2.5),
                array('id' => 'hits', 'name' => 'Hits', 'typical_line' => 2.5),
                array('id' => 'blocked_shots', 'name' => 'Blocked Shots', 'typical_line' => 1.5),
                array('id' => 'power_play_points', 'name' => 'Power Play Points', 'typical_line' => 0.5),
                array('id' => 'faceoff_wins', 'name' => 'Faceoff Wins', 'typical_line' => 10.5)
            )
        );
        
        return isset($prop_types[$sport]) ? $prop_types[$sport] : array();
    }
    
    private function get_cached_data($key) {
        if (!get_option('pbg_enable_caching', true)) {
            return false;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'pbg_cache';
        
        $result = $wpdb->get_row($wpdb->prepare(
            "SELECT cache_value FROM $table_name WHERE cache_key = %s AND expiry_time > NOW()",
            $key
        ));
        
        if ($result) {
            return json_decode($result->cache_value, true);
        }
        
        return false;
    }
    
    private function cache_data($key, $data, $duration = 300) {
        if (!get_option('pbg_enable_caching', true)) {
            return;
        }
        
        global $wpdb;
        $table_name = $wpdb->prefix . 'pbg_cache';
        
        $expiry_time = date('Y-m-d H:i:s', time() + $duration);
        
        $wpdb->replace(
            $table_name,
            array(
                'cache_key' => $key,
                'cache_value' => json_encode($data),
                'expiry_time' => $expiry_time
            ),
            array('%s', '%s', '%s')
        );
    }
}
