<?php
/**
 * Player Service Class for handling player data and stats
 */

if (!defined('ABSPATH')) {
    exit;
}

class PBG_Player_Service {
    
    public function __construct() {
        // Constructor
    }
    
    public function get_players_by_team($sport, $team) {
        $database = $this->get_player_database();
        
        if (!isset($database[$sport]['teams'][$team])) {
            return array();
        }
        
        $team_roster = $database[$sport]['teams'][$team];
        $players = array();
        
        foreach ($team_roster as $position => $player_list) {
            foreach ($player_list as $player) {
                $players[] = array(
                    'name' => $player,
                    'team' => $team,
                    'position' => $position
                );
            }
        }
        
        return $players;
    }
    
    public function get_all_players($sport) {
        $database = $this->get_player_database();
        
        if (!isset($database[$sport]['teams'])) {
            return array();
        }
        
        $all_players = array();
        
        foreach ($database[$sport]['teams'] as $team_name => $positions) {
            foreach ($positions as $position => $player_list) {
                foreach ($player_list as $player) {
                    $all_players[] = array(
                        'name' => $player,
                        'team' => $team_name,
                        'position' => $position
                    );
                }
            }
        }
        
        return $all_players;
    }
    
    public function get_player_stats($sport, $player_name) {
        // Generate realistic mock stats based on sport and player
        $stats = $this->generate_mock_stats($sport, $player_name);
        
        return array(
            'player_name' => $player_name,
            'position' => $this->get_player_position($sport, $player_name),
            'team' => $this->get_player_team($sport, $player_name),
            'season_stats' => $stats,
            'recent_games' => array(),
            'injury_status' => null
        );
    }
    
    private function get_player_position($sport, $player_name) {
        $database = $this->get_player_database();
        
        if (!isset($database[$sport]['teams'])) {
            return 'Unknown';
        }
        
        foreach ($database[$sport]['teams'] as $team_name => $positions) {
            foreach ($positions as $position => $players) {
                if (in_array($player_name, $players)) {
                    return $position;
                }
            }
        }
        
        return 'Unknown';
    }
    
    private function get_player_team($sport, $player_name) {
        $database = $this->get_player_database();
        
        if (!isset($database[$sport]['teams'])) {
            return 'Unknown';
        }
        
        foreach ($database[$sport]['teams'] as $team_name => $positions) {
            foreach ($positions as $position => $players) {
                if (in_array($player_name, $players)) {
                    return $team_name;
                }
            }
        }
        
        return 'Unknown';
    }
    
    private function generate_mock_stats($sport, $player_name) {
        $hash = crc32($player_name);
        $stats = array();
        
        switch ($sport) {
            case 'americanfootball_nfl':
                $position = $this->get_player_position($sport, $player_name);
                if ($position === 'QB') {
                    $stats = array(
                        'passing_yards' => 4200 + ($hash % 800),
                        'passing_touchdowns' => 30 + ($hash % 15),
                        'rushing_yards' => 400 + ($hash % 300),
                        'interceptions' => 8 + ($hash % 8)
                    );
                } else {
                    $stats = array(
                        'rushing_yards' => 1200 + ($hash % 600),
                        'rushing_touchdowns' => 8 + ($hash % 8),
                        'receiving_yards' => 300 + ($hash % 400),
                        'receptions' => 25 + ($hash % 35)
                    );
                }
                break;
                
            case 'basketball_nba':
                $stats = array(
                    'points_per_game' => 20 + ($hash % 15),
                    'rebounds_per_game' => 6 + ($hash % 8),
                    'assists_per_game' => 4 + ($hash % 6),
                    'three_pointers_made' => 150 + ($hash % 100),
                    'field_goal_percentage' => 0.45 + (($hash % 20) / 100)
                );
                break;
                
            case 'baseball_mlb':
                $stats = array(
                    'batting_average' => 0.250 + (($hash % 100) / 1000),
                    'home_runs' => 15 + ($hash % 25),
                    'rbis' => 60 + ($hash % 40),
                    'hits' => 140 + ($hash % 60),
                    'stolen_bases' => 5 + ($hash % 20)
                );
                break;
                
            case 'icehockey_nhl':
                $stats = array(
                    'goals' => 15 + ($hash % 25),
                    'assists' => 20 + ($hash % 30),
                    'points' => 35 + ($hash % 40),
                    'shots_on_goal' => 180 + ($hash % 120),
                    'penalty_minutes' => 20 + ($hash % 40)
                );
                break;
        }
        
        return $stats;
    }
    
    private function get_player_database() {
        return include PBG_PLUGIN_PATH . 'includes/player-database.php';
    }
}
