<?php
/**
 * Frontend Display Template
 */

if (!defined('ABSPATH')) {
    exit;
}

$unique_id = uniqid('pbg_');
$theme_class = 'pbg-theme-' . esc_attr($atts['theme']);
?>

<div id="<?php echo $unique_id; ?>" class="prop-bet-generator <?php echo $theme_class; ?>">
    <div class="pbg-container">
        <!-- Header -->
        <div class="pbg-header">
            <h2 class="pbg-title">🎯 Enhanced Prop Bet Generator</h2>
            <p class="pbg-subtitle">AI-powered prop bet recommendations with real player stats and game selection</p>
            
            <!-- API Status Indicators -->
            <div class="pbg-status-indicators">
                <div class="pbg-status-item" id="pbg-odds-status">
                    <span class="pbg-status-icon">📊</span>
                    <span class="pbg-status-text">Odds API: <span class="pbg-status-value">Checking...</span></span>
                </div>
                <div class="pbg-status-item" id="pbg-ai-status">
                    <span class="pbg-status-icon">🤖</span>
                    <span class="pbg-status-text">AI Engine: <span class="pbg-status-value">Checking...</span></span>
                </div>
            </div>
        </div>
        
        <!-- Main Content -->
        <div class="pbg-main-content">
            <div class="pbg-form-section">
                <h3>Generate Prop Bets</h3>
                
                <form id="pbg-form" class="pbg-form">
                    <!-- Sport Selection -->
                    <div class="pbg-field">
                        <label for="pbg-sport">Select Sport</label>
                        <select id="pbg-sport" name="sport" required>
                            <option value="">Choose a sport...</option>
                        </select>
                    </div>
                    
                    <!-- Selection Mode Toggle -->
                    <div class="pbg-field pbg-selection-mode" style="display: none;">
                        <label>Player Selection Method</label>
                        <div class="pbg-radio-group">
                            <label class="pbg-radio-label">
                                <input type="radio" name="selection_mode" value="game" checked>
                                <span>🎮 Select from Game/Teams</span>
                            </label>
                            <label class="pbg-radio-label">
                                <input type="radio" name="selection_mode" value="manual">
                                <span>✏️ Manual Entry</span>
                            </label>
                        </div>
                    </div>
                    
                    <!-- Game Selection -->
                    <div class="pbg-field pbg-game-selection" style="display: none;">
                        <label for="pbg-game">Select Game</label>
                        <select id="pbg-game" name="game">
                            <option value="">Choose a game...</option>
                        </select>
                    </div>
                    
                    <!-- Player Selection -->
                    <div class="pbg-field pbg-player-selection" style="display: none;">
                        <label for="pbg-player">Select Player</label>
                        <div id="pbg-player-list" class="pbg-player-list">
                            <p class="pbg-loading">Select a game to see available players...</p>
                        </div>
                    </div>
                    
                    <!-- Manual Player Entry -->
                    <div class="pbg-field pbg-manual-entry" style="display: none;">
                        <label for="pbg-player-manual">Player Name</label>
                        <input type="text" id="pbg-player-manual" name="player_manual" 
                               placeholder="Enter player name (e.g., Josh Allen)">
                    </div>
                    
                    <!-- Prop Types -->
                    <div class="pbg-field pbg-prop-types" style="display: none;">
                        <label>Select Prop Types</label>
                        <div id="pbg-prop-list" class="pbg-prop-list">
                            <p class="pbg-loading">Select a sport to see available prop types...</p>
                        </div>
                    </div>
                    
                    <!-- Submit Button -->
                    <div class="pbg-field">
                        <button type="submit" id="pbg-submit" class="pbg-btn pbg-btn-primary" disabled>
                            <span class="pbg-btn-text">🚀 Generate Enhanced Prop Bets</span>
                            <span class="pbg-btn-loading" style="display: none;">
                                <span class="pbg-spinner"></span>
                                Generating AI Recommendations...
                            </span>
                        </button>
                    </div>
                    
                    <!-- Error Display -->
                    <div id="pbg-error" class="pbg-error" style="display: none;"></div>
                </form>
            </div>
            
            <!-- Results Section -->
            <div class="pbg-results-section">
                <h3>AI Recommendations</h3>
                
                <div id="pbg-results" class="pbg-results">
                    <div class="pbg-results-empty">
                        <div class="pbg-empty-icon">🎯</div>
                        <p>Select sport, game, player, and prop types to generate AI-powered recommendations</p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Footer -->
        <div class="pbg-footer">
            <p>Enhanced with Game Selection & Player Rosters • Powered by TheOddsAPI & OpenAI</p>
            <p class="pbg-disclaimer">Please gamble responsibly • For educational purposes only</p>
        </div>
    </div>
</div>

<script>
(function() {
    const pbg = {
        containerId: '<?php echo $unique_id; ?>',
        container: null,
        data: {
            sports: [],
            games: [],
            players: [],
            propTypes: [],
            selectedSport: '',
            selectedGame: '',
            selectedPlayer: '',
            selectionMode: 'game'
        },
        
        init: function() {
            this.container = document.getElementById(this.containerId);
            this.bindEvents();
            this.loadSports();
            this.checkApiStatus();
        },
        
        bindEvents: function() {
            const form = this.container.querySelector('#pbg-form');
            const sportSelect = this.container.querySelector('#pbg-sport');
            const gameSelect = this.container.querySelector('#pbg-game');
            const modeRadios = this.container.querySelectorAll('input[name="selection_mode"]');
            
            form.addEventListener('submit', this.handleSubmit.bind(this));
            sportSelect.addEventListener('change', this.handleSportChange.bind(this));
            gameSelect.addEventListener('change', this.handleGameChange.bind(this));
            
            modeRadios.forEach(radio => {
                radio.addEventListener('change', this.handleModeChange.bind(this));
            });
        },
        
        loadSports: function() {
            this.ajaxRequest('pbg_get_sports', {}, (response) => {
                if (response.success) {
                    this.data.sports = response.data;
                    this.populateSports();
                }
            });
        },
        
        populateSports: function() {
            const select = this.container.querySelector('#pbg-sport');
            select.innerHTML = '<option value="">Choose a sport...</option>';
            
            this.data.sports.forEach(sport => {
                const option = document.createElement('option');
                option.value = sport.key;
                option.textContent = this.getSportDisplayName(sport.key);
                select.appendChild(option);
            });
        },
        
        handleSportChange: function(e) {
            this.data.selectedSport = e.target.value;
            
            if (this.data.selectedSport) {
                this.container.querySelector('.pbg-selection-mode').style.display = 'block';
                this.loadGames();
                this.loadPropTypes();
            } else {
                this.hideGameElements();
            }
        },
        
        loadGames: function() {
            this.ajaxRequest('pbg_get_games', {
                sport: this.data.selectedSport
            }, (response) => {
                if (response.success) {
                    this.data.games = response.data;
                    this.populateGames();
                }
            });
        },
        
        populateGames: function() {
            const select = this.container.querySelector('#pbg-game');
            select.innerHTML = '<option value="">Choose a game...</option>';
            
            this.data.games.forEach((game, index) => {
                const option = document.createElement('option');
                option.value = index;
                option.textContent = `${game.away_team} @ ${game.home_team}`;
                if (game.commence_time) {
                    option.textContent += ` - ${new Date(game.commence_time).toLocaleDateString()}`;
                }
                select.appendChild(option);
            });
            
            this.container.querySelector('.pbg-game-selection').style.display = 'block';
        },
        
        handleGameChange: function(e) {
            const gameIndex = parseInt(e.target.value);
            if (!isNaN(gameIndex) && this.data.games[gameIndex]) {
                this.data.selectedGame = gameIndex;
                this.data.players = this.data.games[gameIndex].all_players || [];
                this.populatePlayers();
            }
        },
        
        populatePlayers: function() {
            const container = this.container.querySelector('#pbg-player-list');
            
            if (this.data.players.length === 0) {
                container.innerHTML = '<p class="pbg-loading">No players available for this game.</p>';
                return;
            }
            
            container.innerHTML = '';
            
            this.data.players.forEach(player => {
                const button = document.createElement('button');
                button.type = 'button';
                button.className = 'pbg-player-btn';
                button.innerHTML = `
                    <span class="pbg-player-name">${player.name}</span>
                    <div class="pbg-player-info">
                        <span class="pbg-player-position">${player.position}</span>
                        <span class="pbg-player-team">${player.team}</span>
                    </div>
                `;
                
                button.addEventListener('click', () => {
                    this.selectPlayer(player.name, button);
                });
                
                container.appendChild(button);
            });
            
            this.container.querySelector('.pbg-player-selection').style.display = 'block';
        },
        
        selectPlayer: function(playerName, buttonElement) {
            // Remove previous selection
            this.container.querySelectorAll('.pbg-player-btn').forEach(btn => {
                btn.classList.remove('pbg-selected');
            });
            
            // Add selection to clicked button
            buttonElement.classList.add('pbg-selected');
            this.data.selectedPlayer = playerName;
            
            this.updateSubmitButton();
        },
        
        loadPropTypes: function() {
            const sport = this.data.sports.find(s => s.key === this.data.selectedSport);
            if (sport && sport.prop_types) {
                this.data.propTypes = sport.prop_types;
                this.populatePropTypes();
            }
        },
        
        populatePropTypes: function() {
            const container = this.container.querySelector('#pbg-prop-list');
            container.innerHTML = '';
            
            this.data.propTypes.forEach(prop => {
                const label = document.createElement('label');
                label.className = 'pbg-prop-label';
                label.innerHTML = `
                    <input type="checkbox" name="prop_types[]" value="${prop.id}">
                    <span class="pbg-prop-name">${prop.name}</span>
                    <span class="pbg-prop-line">Typical: ${prop.typical_line}</span>
                `;
                container.appendChild(label);
            });
            
            // Auto-select first 3 props
            const checkboxes = container.querySelectorAll('input[type="checkbox"]');
            for (let i = 0; i < Math.min(3, checkboxes.length); i++) {
                checkboxes[i].checked = true;
            }
            
            this.container.querySelector('.pbg-prop-types').style.display = 'block';
            this.updateSubmitButton();
        },
        
        handleModeChange: function(e) {
            this.data.selectionMode = e.target.value;
            
            if (this.data.selectionMode === 'game') {
                this.container.querySelector('.pbg-game-selection').style.display = 'block';
                this.container.querySelector('.pbg-player-selection').style.display = 'block';
                this.container.querySelector('.pbg-manual-entry').style.display = 'none';
            } else {
                this.container.querySelector('.pbg-game-selection').style.display = 'none';
                this.container.querySelector('.pbg-player-selection').style.display = 'none';
                this.container.querySelector('.pbg-manual-entry').style.display = 'block';
            }
            
            this.updateSubmitButton();
        },
        
        updateSubmitButton: function() {
            const button = this.container.querySelector('#pbg-submit');
            const sport = this.data.selectedSport;
            const player = this.data.selectionMode === 'game' ? 
                this.data.selectedPlayer : 
                this.container.querySelector('#pbg-player-manual').value.trim();
            const props = this.container.querySelectorAll('input[name="prop_types[]"]:checked');
            
            button.disabled = !sport || !player || props.length === 0;
        },
        
        handleSubmit: function(e) {
            e.preventDefault();
            
            const button = this.container.querySelector('#pbg-submit');
            const buttonText = button.querySelector('.pbg-btn-text');
            const buttonLoading = button.querySelector('.pbg-btn-loading');
            
            // Get form data
            const playerName = this.data.selectionMode === 'game' ? 
                this.data.selectedPlayer : 
                this.container.querySelector('#pbg-player-manual').value.trim();
            
            const propTypes = Array.from(this.container.querySelectorAll('input[name="prop_types[]"]:checked'))
                .map(cb => cb.value);
            
            // Show loading state
            button.disabled = true;
            buttonText.style.display = 'none';
            buttonLoading.style.display = 'inline-flex';
            
            // Clear previous results
            this.container.querySelector('#pbg-results').innerHTML = 
                '<div class="pbg-loading-results"><span class="pbg-spinner"></span> Generating AI recommendations...</div>';
            
            this.ajaxRequest('pbg_generate_props', {
                sport: this.data.selectedSport,
                player_name: playerName,
                prop_types: propTypes,
                game_id: this.data.selectedGame
            }, (response) => {
                // Reset button state
                button.disabled = false;
                buttonText.style.display = 'inline';
                buttonLoading.style.display = 'none';
                
                if (response.success) {
                    this.displayResults(response.data);
                } else {
                    this.showError('Failed to generate recommendations: ' + response.data);
                }
            });
        },
        
        displayResults: function(data) {
            const container = this.container.querySelector('#pbg-results');
            
            if (!data.recommendations || data.recommendations.length === 0) {
                container.innerHTML = '<div class="pbg-results-empty"><p>No recommendations generated.</p></div>';
                return;
            }
            
            let html = '';
            
            data.recommendations.forEach(rec => {
                html += `
                    <div class="pbg-recommendation">
                        <div class="pbg-rec-header">
                            <div class="pbg-rec-player">
                                <h4>${rec.player_name}</h4>
                                <p>${rec.prop_display_name}</p>
                            </div>
                            <div class="pbg-rec-confidence pbg-confidence-${rec.confidence.toLowerCase()}">
                                ${rec.confidence} Confidence
                            </div>
                        </div>
                        
                        <div class="pbg-rec-stats">
                            <div class="pbg-stat-box">
                                <label>Recommended Line</label>
                                <span class="pbg-stat-value">${rec.recommended_line}</span>
                            </div>
                            ${rec.best_odds ? `
                                <div class="pbg-stat-box">
                                    <label>Best Odds</label>
                                    <span class="pbg-stat-value pbg-odds-value">${rec.best_odds}</span>
                                    ${rec.sportsbook ? `<small>${rec.sportsbook}</small>` : ''}
                                </div>
                            ` : ''}
                        </div>
                        
                        <div class="pbg-rec-explanation">
                            <label>🧠 AI Analysis</label>
                            <p>${rec.explanation}</p>
                        </div>
                    </div>
                `;
            });
            
            container.innerHTML = html;
        },
        
        showError: function(message) {
            const errorDiv = this.container.querySelector('#pbg-error');
            errorDiv.textContent = message;
            errorDiv.style.display = 'block';
            
            setTimeout(() => {
                errorDiv.style.display = 'none';
            }, 5000);
        },
        
        checkApiStatus: function() {
            // This would typically check API status
            // For now, we'll assume they're working
            this.container.querySelector('#pbg-odds-status .pbg-status-value').textContent = 'Connected';
            this.container.querySelector('#pbg-ai-status .pbg-status-value').textContent = 'Connected';
        },
        
        getSportDisplayName: function(sportKey) {
            const names = {
                'americanfootball_nfl': 'NFL',
                'basketball_nba': 'NBA',
                'baseball_mlb': 'MLB',
                'icehockey_nhl': 'NHL'
            };
            return names[sportKey] || sportKey;
        },
        
        hideGameElements: function() {
            this.container.querySelector('.pbg-selection-mode').style.display = 'none';
            this.container.querySelector('.pbg-game-selection').style.display = 'none';
            this.container.querySelector('.pbg-player-selection').style.display = 'none';
            this.container.querySelector('.pbg-manual-entry').style.display = 'none';
            this.container.querySelector('.pbg-prop-types').style.display = 'none';
        },
        
        ajaxRequest: function(action, data, callback) {
            const formData = new FormData();
            formData.append('action', action);
            formData.append('nonce', pbg_ajax.nonce);
            
            for (const key in data) {
                if (Array.isArray(data[key])) {
                    data[key].forEach(value => {
                        formData.append(key + '[]', value);
                    });
                } else {
                    formData.append(key, data[key]);
                }
            }
            
            fetch(pbg_ajax.ajax_url, {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(callback)
            .catch(error => {
                console.error('AJAX error:', error);
                this.showError('Network error occurred');
            });
        }
    };
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => pbg.init());
    } else {
        pbg.init();
    }
})();
</script>
