<?php
/**
 * Settings Page Template
 */

if (!defined('ABSPATH')) {
    exit;
}

// Handle form submission
if (isset($_POST['submit']) && check_admin_referer('pbg_settings_nonce')) {
    update_option('pbg_odds_api_key', sanitize_text_field($_POST['pbg_odds_api_key']));
    update_option('pbg_openai_api_key', sanitize_text_field($_POST['pbg_openai_api_key']));
    update_option('pbg_enable_caching', isset($_POST['pbg_enable_caching']));
    update_option('pbg_cache_duration', intval($_POST['pbg_cache_duration']));
    
    echo '<div class="notice notice-success"><p>Settings saved successfully!</p></div>';
}

// Get current settings
$odds_api_key = get_option('pbg_odds_api_key', '');
$openai_api_key = get_option('pbg_openai_api_key', '');
$enable_caching = get_option('pbg_enable_caching', true);
$cache_duration = get_option('pbg_cache_duration', 300);
?>

<div class="wrap">
    <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
    
    <div class="pbg-settings-container">
        <form method="post" action="">
            <?php wp_nonce_field('pbg_settings_nonce'); ?>
            
            <div class="pbg-settings-section">
                <h2>🔑 API Configuration</h2>
                <p>Configure your API keys for TheOddsAPI and OpenAI to enable full functionality.</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="pbg_odds_api_key">TheOddsAPI Key</label>
                        </th>
                        <td>
                            <input type="text" id="pbg_odds_api_key" name="pbg_odds_api_key" 
                                   value="<?php echo esc_attr($odds_api_key); ?>" 
                                   class="regular-text" />
                            <p class="description">
                                Get your free API key from <a href="https://the-odds-api.com/" target="_blank">TheOddsAPI</a>. 
                                Free tier includes 500 requests per month.
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="pbg_openai_api_key">OpenAI API Key</label>
                        </th>
                        <td>
                            <input type="password" id="pbg_openai_api_key" name="pbg_openai_api_key" 
                                   value="<?php echo esc_attr($openai_api_key); ?>" 
                                   class="regular-text" />
                            <p class="description">
                                Get your API key from <a href="https://platform.openai.com/api-keys" target="_blank">OpenAI</a>. 
                                Required for AI-powered prop bet analysis.
                            </p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <div class="pbg-settings-section">
                <h2>⚡ Performance Settings</h2>
                <p>Configure caching and performance options.</p>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable Caching</th>
                        <td>
                            <label>
                                <input type="checkbox" name="pbg_enable_caching" value="1" 
                                       <?php checked($enable_caching); ?> />
                                Enable API response caching
                            </label>
                            <p class="description">
                                Reduces API calls and improves performance by caching responses.
                            </p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="pbg_cache_duration">Cache Duration (seconds)</label>
                        </th>
                        <td>
                            <input type="number" id="pbg_cache_duration" name="pbg_cache_duration" 
                                   value="<?php echo esc_attr($cache_duration); ?>" 
                                   min="60" max="3600" class="small-text" />
                            <p class="description">
                                How long to cache API responses (60-3600 seconds). Default: 300 (5 minutes).
                            </p>
                        </td>
                    </tr>
                </table>
            </div>
            
            <div class="pbg-settings-section">
                <h2>🧪 API Test</h2>
                <p>Test your API connections to ensure everything is working correctly.</p>
                
                <div class="pbg-api-test">
                    <button type="button" id="pbg-test-apis" class="button button-secondary">
                        Test API Connections
                    </button>
                    <div id="pbg-test-results" style="margin-top: 15px;"></div>
                </div>
            </div>
            
            <?php submit_button('Save Settings'); ?>
        </form>
    </div>
</div>

<style>
.pbg-settings-container {
    max-width: 800px;
    margin: 20px 0;
}

.pbg-settings-section {
    background: white;
    border: 1px solid #ddd;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.pbg-settings-section h2 {
    margin-top: 0;
    color: #333;
    border-bottom: 2px solid #667eea;
    padding-bottom: 10px;
}

.pbg-api-test {
    padding: 15px;
    background: #f8f9fa;
    border-radius: 5px;
    border: 1px solid #ddd;
}

#pbg-test-results {
    display: none;
}

.pbg-test-result {
    padding: 10px;
    margin: 5px 0;
    border-radius: 4px;
}

.pbg-test-success {
    background: #d4edda;
    border: 1px solid #c3e6cb;
    color: #155724;
}

.pbg-test-error {
    background: #f8d7da;
    border: 1px solid #f5c6cb;
    color: #721c24;
}
</style>

<script>
jQuery(document).ready(function($) {
    $('#pbg-test-apis').on('click', function() {
        var button = $(this);
        var results = $('#pbg-test-results');
        
        button.prop('disabled', true).text('Testing...');
        results.show().html('<p>Testing API connections...</p>');
        
        $.ajax({
            url: ajaxurl,
            method: 'POST',
            data: {
                action: 'pbg_test_apis',
                nonce: '<?php echo wp_create_nonce('pbg_test_nonce'); ?>'
            },
            success: function(response) {
                if (response.success) {
                    var html = '<h4>API Test Results:</h4>';
                    
                    if (response.data.odds_api) {
                        html += '<div class="pbg-test-result pbg-test-success">✅ TheOddsAPI: Connected</div>';
                    } else {
                        html += '<div class="pbg-test-result pbg-test-error">❌ TheOddsAPI: Failed</div>';
                    }
                    
                    if (response.data.openai_api) {
                        html += '<div class="pbg-test-result pbg-test-success">✅ OpenAI API: Connected</div>';
                    } else {
                        html += '<div class="pbg-test-result pbg-test-error">❌ OpenAI API: Failed</div>';
                    }
                    
                    if (response.data.errors && response.data.errors.length > 0) {
                        html += '<h4>Errors:</h4>';
                        response.data.errors.forEach(function(error) {
                            html += '<div class="pbg-test-result pbg-test-error">' + error + '</div>';
                        });
                    }
                    
                    results.html(html);
                } else {
                    results.html('<div class="pbg-test-result pbg-test-error">Test failed: ' + response.data + '</div>');
                }
            },
            error: function() {
                results.html('<div class="pbg-test-result pbg-test-error">Test failed: Network error</div>');
            },
            complete: function() {
                button.prop('disabled', false).text('Test API Connections');
            }
        });
    });
});
</script>
