<?php
/**
 * Plugin Name: Enhanced Prop Bet Generator
 * Plugin URI: https://gamblingsite.com/
 * Description: AI-powered prop bet recommendations with real player stats, game selection, and multi-sport support. Integrates TheOddsAPI and OpenAI for professional betting insights.
 * Version: 2.0.2
 * Author: GamblingSite.com
 * Author URI: https://gamblingsite.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: prop-bet-generator
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('PBG_PLUGIN_URL', plugin_dir_url(__FILE__));
define('PBG_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('PBG_VERSION', '2.0.2');

// Main Plugin Class
class PropBetGenerator {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('wp_ajax_pbg_generate_props', array($this, 'ajax_generate_props'));
        add_action('wp_ajax_nopriv_pbg_generate_props', array($this, 'ajax_generate_props'));
        add_action('wp_ajax_pbg_get_sports', array($this, 'ajax_get_sports'));
        add_action('wp_ajax_nopriv_pbg_get_sports', array($this, 'ajax_get_sports'));
        add_action('wp_ajax_pbg_get_games', array($this, 'ajax_get_games'));
        add_action('wp_ajax_nopriv_pbg_get_games', array($this, 'ajax_get_games'));
        add_action('wp_ajax_pbg_get_players', array($this, 'ajax_get_players'));
        add_action('wp_ajax_nopriv_pbg_get_players', array($this, 'ajax_get_players'));
        add_action('wp_ajax_pbg_test_apis', array($this, 'ajax_test_apis'));
        add_action('wp_ajax_nopriv_pbg_test_apis', array($this, 'ajax_test_apis'));
        add_shortcode('prop_bet_generator', array($this, 'shortcode_display'));
        
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function init() {
        load_plugin_textdomain('prop-bet-generator', false, dirname(plugin_basename(__FILE__)) . '/languages');
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('pbg-main', PBG_PLUGIN_URL . 'assets/js/prop-bet-generator.js', array('jquery'), PBG_VERSION, true);
        wp_enqueue_style('pbg-style', PBG_PLUGIN_URL . 'assets/css/prop-bet-generator.css', array(), PBG_VERSION);
        
        // Localize script for AJAX
        wp_localize_script('pbg-main', 'pbg_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('pbg_nonce'),
        ));
    }
    
    public function add_admin_menu() {
        add_menu_page(
            'Prop Bet Generator',
            'Prop Bet Generator',
            'manage_options',
            'prop-bet-generator',
            array($this, 'admin_page'),
            'dashicons-chart-line',
            30
        );
        
        add_submenu_page(
            'prop-bet-generator',
            'Settings',
            'Settings',
            'manage_options',
            'prop-bet-generator-settings',
            array($this, 'settings_page')
        );
    }
    
    public function admin_page() {
        include PBG_PLUGIN_PATH . 'includes/admin-page.php';
    }
    
    public function settings_page() {
        include PBG_PLUGIN_PATH . 'includes/settings-page.php';
    }
    
    public function shortcode_display($atts) {
        $atts = shortcode_atts(array(
            'theme' => 'dark',
            'sports' => 'all',
            'default_sport' => 'americanfootball_nfl'
        ), $atts);
        
        ob_start();
        include PBG_PLUGIN_PATH . 'includes/frontend-display.php';
        return ob_get_clean();
    }
    
    public function ajax_get_sports() {
        check_ajax_referer('pbg_nonce', 'nonce');
        
        $api_service = new PBG_API_Service();
        $sports = $api_service->get_sports();
        
        wp_send_json_success($sports);
    }
    
    public function ajax_get_games() {
        check_ajax_referer('pbg_nonce', 'nonce');
        
        $sport = sanitize_text_field($_POST['sport']);
        $api_service = new PBG_API_Service();
        $games = $api_service->get_enhanced_games($sport);
        
        wp_send_json_success($games);
    }
    
    public function ajax_get_players() {
        check_ajax_referer('pbg_nonce', 'nonce');
        
        $sport = sanitize_text_field($_POST['sport']);
        $team = sanitize_text_field($_POST['team']);
        
        $player_service = new PBG_Player_Service();
        $players = $player_service->get_players_by_team($sport, $team);
        
        wp_send_json_success($players);
    }
    
    public function ajax_generate_props() {
        check_ajax_referer('pbg_nonce', 'nonce');
        
        $sport = sanitize_text_field($_POST['sport']);
        $player_name = sanitize_text_field($_POST['player_name']);
        $prop_types = array_map('sanitize_text_field', $_POST['prop_types']);
        $game_id = sanitize_text_field($_POST['game_id']);
        
        $ai_service = new PBG_AI_Service();
        $player_service = new PBG_Player_Service();
        
        // Get player stats
        $player_stats = $player_service->get_player_stats($sport, $player_name);
        
        // Generate recommendations
        $recommendations = $ai_service->generate_recommendations($player_stats, $sport, $prop_types);
        
        wp_send_json_success(array(
            'recommendations' => $recommendations,
            'player_stats' => $player_stats
        ));
    }
    
    public function ajax_test_apis() {
        check_ajax_referer('pbg_test_nonce', 'nonce');
        
        $results = array(
            'odds_api' => false,
            'openai_api' => false,
            'errors' => array()
        );
        
        // Clear cache before testing
        $this->clear_cache();
        
        // Test TheOddsAPI
        try {
            $api_service = new PBG_API_Service();
            $sports = $api_service->get_sports();
            if (!empty($sports)) {
                $results['odds_api'] = true;
            } else {
                $results['errors'][] = 'TheOddsAPI returned no sports data';
            }
        } catch (Exception $e) {
            $results['errors'][] = 'TheOddsAPI error: ' . $e->getMessage();
        }
        
        // Test OpenAI
        try {
            $ai_service = new PBG_AI_Service();
            $mock_player = array(
                'player_name' => 'Test Player',
                'position' => 'QB',
                'team' => 'Test Team',
                'season_stats' => array('passing_yards' => 3000)
            );
            
            $recommendations = $ai_service->generate_recommendations($mock_player, 'americanfootball_nfl', array('passing_yards'));
            if (!empty($recommendations)) {
                $results['openai_api'] = true;
            } else {
                $results['errors'][] = 'OpenAI API returned no recommendations';
            }
        } catch (Exception $e) {
            $results['errors'][] = 'OpenAI API error: ' . $e->getMessage();
        }
        
        wp_send_json_success($results);
    }
    
    private function clear_cache() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'pbg_cache';
        $wpdb->query("DELETE FROM $table_name WHERE cache_key LIKE 'pbg_%'");
    }
    
    public function activate() {
        // Create database tables if needed
        $this->create_tables();
        
        // Set default options
        add_option('pbg_odds_api_key', '');
        add_option('pbg_openai_api_key', '');
        add_option('pbg_enable_caching', true);
        add_option('pbg_cache_duration', 300); // 5 minutes
    }
    
    public function deactivate() {
        // Clean up scheduled events
        wp_clear_scheduled_hook('pbg_clear_cache');
    }
    
    private function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'pbg_cache';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            cache_key varchar(255) NOT NULL,
            cache_value longtext NOT NULL,
            expiry_time datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            UNIQUE KEY cache_key (cache_key)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
}

// Include required files
require_once PBG_PLUGIN_PATH . 'includes/class-api-service.php';
require_once PBG_PLUGIN_PATH . 'includes/class-player-service.php';
require_once PBG_PLUGIN_PATH . 'includes/class-ai-service.php';
require_once PBG_PLUGIN_PATH . 'includes/player-database.php';

// Initialize the plugin
new PropBetGenerator();
