<?php
/**
 * Plugin Name: AI Sports Betting Picks
 * Plugin URI: https://gamblingSite.com
 * Description: AI-powered sports betting analysis and recommendations with real-time odds from top sportsbooks.
 * Version: 1.0.0
 * Author: GamblingSite.com
 * License: GPL v2 or later
 * Text Domain: ai-sports-betting-picks
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('ASBP_PLUGIN_URL', plugin_dir_url(__FILE__));
define('ASBP_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('ASBP_VERSION', '1.0.0');

// Main plugin class
class AISportsBettingPicks {
    
    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('admin_menu', array($this, 'admin_menu'));
        add_action('wp_ajax_asbp_get_pick_of_day', array($this, 'ajax_get_pick_of_day'));
        add_action('wp_ajax_nopriv_asbp_get_pick_of_day', array($this, 'ajax_get_pick_of_day'));
        add_action('wp_ajax_asbp_get_sports', array($this, 'ajax_get_sports'));
        add_action('wp_ajax_nopriv_asbp_get_sports', array($this, 'ajax_get_sports'));
        add_action('wp_ajax_asbp_get_games', array($this, 'ajax_get_games'));
        add_action('wp_ajax_nopriv_asbp_get_games', array($this, 'ajax_get_games'));
        add_action('wp_ajax_asbp_generate_pick', array($this, 'ajax_generate_pick'));
        add_action('wp_ajax_nopriv_asbp_generate_pick', array($this, 'ajax_generate_pick'));
        
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    public function init() {
        // Add shortcodes
        add_shortcode('ai_sports_picks', array($this, 'shortcode_display'));
        add_shortcode('ai_pick_of_day', array($this, 'shortcode_pick_of_day'));
        
        // Create database tables if needed
        $this->create_tables();
    }
    
    public function enqueue_scripts() {
        wp_enqueue_script('asbp-main', ASBP_PLUGIN_URL . 'assets/js/main.js', array('jquery'), ASBP_VERSION, true);
        wp_enqueue_style('asbp-styles', ASBP_PLUGIN_URL . 'assets/css/styles.css', array(), ASBP_VERSION);
        
        // Localize script for AJAX
        wp_localize_script('asbp-main', 'asbp_ajax', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('asbp_nonce')
        ));
    }
    
    public function admin_menu() {
        add_menu_page(
            'AI Sports Betting Picks',
            'Sports Picks',
            'manage_options',
            'ai-sports-betting-picks',
            array($this, 'admin_page'),
            'dashicons-chart-line',
            30
        );
        
        add_submenu_page(
            'ai-sports-betting-picks',
            'Settings',
            'Settings',
            'manage_options',
            'asbp-settings',
            array($this, 'settings_page')
        );
    }
    
    public function admin_page() {
        include ASBP_PLUGIN_PATH . 'admin/admin-page.php';
    }
    
    public function settings_page() {
        if (isset($_POST['submit'])) {
            update_option('asbp_odds_api_key', sanitize_text_field($_POST['odds_api_key']));
            update_option('asbp_openai_api_key', sanitize_text_field($_POST['openai_api_key']));
            echo '<div class="notice notice-success"><p>Settings saved!</p></div>';
        }
        include ASBP_PLUGIN_PATH . 'admin/settings-page.php';
    }
    
    public function shortcode_display($atts) {
        $atts = shortcode_atts(array(
            'show_pick_of_day' => 'true',
            'show_sports_selector' => 'true',
            'theme' => 'dark'
        ), $atts);
        
        ob_start();
        include ASBP_PLUGIN_PATH . 'templates/main-display.php';
        return ob_get_clean();
    }
    
    public function shortcode_pick_of_day($atts) {
        ob_start();
        include ASBP_PLUGIN_PATH . 'templates/pick-of-day.php';
        return ob_get_clean();
    }
    
    // AJAX Handlers
    public function ajax_get_pick_of_day() {
        check_ajax_referer('asbp_nonce', 'nonce');
        
        $pick_of_day = $this->get_pick_of_day();
        wp_send_json_success($pick_of_day);
    }
    
    public function ajax_get_sports() {
        check_ajax_referer('asbp_nonce', 'nonce');
        
        $sports = array('NFL', 'NBA', 'MLB', 'NHL', 'MLS');
        wp_send_json_success($sports);
    }
    
    public function ajax_get_games() {
        check_ajax_referer('asbp_nonce', 'nonce');
        
        $sport = sanitize_text_field($_POST['sport']);
        $games = $this->get_games_by_sport($sport);
        wp_send_json_success($games);
    }
    
    public function ajax_generate_pick() {
        check_ajax_referer('asbp_nonce', 'nonce');
        
        $game_id = sanitize_text_field($_POST['game_id']);
        $sport = sanitize_text_field($_POST['sport']);
        
        $pick = $this->generate_ai_pick($game_id, $sport);
        wp_send_json_success($pick);
    }
    
    // Core functionality methods
    private function get_pick_of_day() {
        // Check if we have a cached pick for today
        $today = date('Y-m-d');
        $cached_pick = get_transient('asbp_pick_of_day_' . $today);
        
        if ($cached_pick !== false) {
            return $cached_pick;
        }
        
        // Generate new pick of the day
        $sports = array('americanfootball_nfl', 'basketball_nba', 'baseball_mlb', 'icehockey_nhl', 'soccer_usa_mls');
        
        foreach ($sports as $sport) {
            $games = $this->fetch_odds_data($sport);
            
            if (!empty($games)) {
                $today_tomorrow_games = array_filter($games, function($game) {
                    return $this->is_game_today_or_tomorrow($game['commence_time']);
                });
                
                if (!empty($today_tomorrow_games)) {
                    usort($today_tomorrow_games, function($a, $b) {
                        return strtotime($a['commence_time']) - strtotime($b['commence_time']);
                    });
                    
                    $featured_game = $today_tomorrow_games[0];
                    $pick = $this->generate_ai_pick_from_game_data($featured_game);
                    $pick['is_pick_of_day'] = true;
                    
                    $pick_of_day = array(
                        'pick' => $pick,
                        'featured_reason' => 'Selected as today\'s top value bet from games happening today or tomorrow'
                    );
                    
                    // Cache for 1 hour
                    set_transient('asbp_pick_of_day_' . $today, $pick_of_day, HOUR_IN_SECONDS);
                    
                    // Store in database
                    $this->store_pick($pick);
                    
                    return $pick_of_day;
                }
            }
        }
        
        return array('error' => 'No games available today or tomorrow');
    }
    
    private function get_games_by_sport($sport) {
        $sport_map = array(
            'NFL' => 'americanfootball_nfl',
            'NBA' => 'basketball_nba',
            'MLB' => 'baseball_mlb',
            'NHL' => 'icehockey_nhl',
            'MLS' => 'soccer_usa_mls'
        );
        
        if (!isset($sport_map[$sport])) {
            return array('error' => 'Invalid sport');
        }
        
        $games = $this->fetch_odds_data($sport_map[$sport]);
        
        return array('games' => $games);
    }
    
    private function generate_ai_pick($game_id, $sport) {
        $sport_map = array(
            'NFL' => 'americanfootball_nfl',
            'NBA' => 'basketball_nba',
            'MLB' => 'baseball_mlb',
            'NHL' => 'icehockey_nhl',
            'MLS' => 'soccer_usa_mls'
        );
        
        if (!isset($sport_map[$sport])) {
            return array('error' => 'Invalid sport');
        }
        
        $games = $this->fetch_odds_data($sport_map[$sport]);
        
        $game_data = null;
        foreach ($games as $game) {
            if ($game['id'] === $game_id) {
                $game_data = $game;
                break;
            }
        }
        
        if (!$game_data) {
            return array('error' => 'Game not found');
        }
        
        $pick = $this->generate_ai_pick_from_game_data($game_data);
        
        // Store in database
        $this->store_pick($pick);
        
        return array('pick' => $pick);
    }
    
    private function fetch_odds_data($sport) {
        $api_key = get_option('asbp_odds_api_key');
        
        if (empty($api_key)) {
            return array();
        }
        
        $url = "https://api.the-odds-api.com/v4/sports/{$sport}/odds";
        $params = array(
            'apiKey' => $api_key,
            'regions' => 'us',
            'markets' => 'h2h,spreads,totals',
            'oddsFormat' => 'decimal',
            'dateFormat' => 'iso'
        );
        
        $response = wp_remote_get($url . '?' . http_build_query($params), array(
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return array();
        }
        
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        return is_array($data) ? $data : array();
    }
    
    private function generate_ai_pick_from_game_data($game_data) {
        $openai_api_key = get_option('asbp_openai_api_key');
        
        if (empty($openai_api_key)) {
            return array('error' => 'OpenAI API key not configured');
        }
        
        // Prepare game information for AI analysis
        $game_info = sprintf(
            "Game: %s vs %s\nSport: %s\nStart Time: %s\n\nAvailable Odds:\n",
            $game_data['home_team'],
            $game_data['away_team'],
            $game_data['sport_title'],
            $game_data['commence_time']
        );
        
        // Extract odds information from preferred sportsbooks
        $preferred_sbooks = array('betmgm', 'fanduel', 'draftkings', 'caesars', 'espnbet');
        $sportsbook_names = array(
            'betmgm' => 'BetMGM',
            'fanduel' => 'FanDuel',
            'draftkings' => 'DraftKings',
            'caesars' => 'Caesars Sportsbook',
            'espnbet' => 'ESPN BET'
        );
        
        foreach ($game_data['bookmakers'] as $bookmaker) {
            if (in_array($bookmaker['key'], $preferred_sbooks)) {
                $game_info .= "\n" . $bookmaker['title'] . ":\n";
                foreach ($bookmaker['markets'] as $market) {
                    $game_info .= "  " . $market['key'] . ": ";
                    foreach ($market['outcomes'] as $outcome) {
                        $game_info .= $outcome['name'] . " (" . $outcome['price'] . ") ";
                    }
                    $game_info .= "\n";
                }
            }
        }
        
        // Call OpenAI API
        $prompt = "Analyze this game and provide a betting recommendation:\n\n" . $game_info;
        
        $ai_response = $this->call_openai_api($prompt);
        
        // Parse AI response
        $pick_type = 'moneyline';
        $recommendation = $game_data['home_team'];
        $confidence_level = 0.75;
        $analysis = $ai_response;
        
        // Simple parsing logic
        $lines = explode("\n", $ai_response);
        foreach ($lines as $line) {
            $line = trim($line);
            if (stripos($line, 'PICK_TYPE:') === 0) {
                $pick_type = trim(str_replace('PICK_TYPE:', '', $line));
            } elseif (stripos($line, 'RECOMMENDATION:') === 0) {
                $recommendation = trim(str_replace('RECOMMENDATION:', '', $line));
            } elseif (stripos($line, 'CONFIDENCE:') === 0) {
                $conf_str = trim(str_replace('CONFIDENCE:', '', $line));
                $confidence_level = max(0.1, min(1.0, floatval($conf_str)));
            } elseif (stripos($line, 'ANALYSIS:') === 0) {
                $analysis = trim(str_replace('ANALYSIS:', '', $line));
            }
        }
        
        // Get odds and sportsbook info
        $odds_info = $this->get_best_odds_from_preferred_sportsbooks($game_data['bookmakers'], $recommendation, $pick_type);
        
        return array(
            'id' => uniqid(),
            'game_id' => $game_data['id'],
            'sport_key' => $game_data['sport_key'],
            'home_team' => $game_data['home_team'],
            'away_team' => $game_data['away_team'],
            'commence_time' => $game_data['commence_time'],
            'pick_type' => $pick_type,
            'recommendation' => $recommendation,
            'confidence_level' => $confidence_level,
            'odds' => $odds_info['american_odds'],
            'sportsbook' => $odds_info['sportsbook'],
            'sportsbook_display_name' => $odds_info['sportsbook_display_name'],
            'analysis' => $analysis,
            'created_at' => current_time('mysql'),
            'is_pick_of_day' => false
        );
    }
    
    private function call_openai_api($prompt) {
        $api_key = get_option('asbp_openai_api_key');
        
        $headers = array(
            'Authorization' => 'Bearer ' . $api_key,
            'Content-Type' => 'application/json',
        );
        
        $body = json_encode(array(
            'model' => 'gpt-4',
            'messages' => array(
                array(
                    'role' => 'system',
                    'content' => 'You are an expert sports betting analyst. Analyze the provided game data and betting odds to generate intelligent betting recommendations.

Provide a clear, professional analysis with your betting recommendation. Do NOT use JSON format or code blocks.

Your response should be in this exact format:
PICK_TYPE: [moneyline/spread/total]
RECOMMENDATION: [specific team name for moneyline, or over/under for totals, or team name with spread for spreads]
CONFIDENCE: [number between 0.1 and 1.0]
ANALYSIS: [2-3 sentences explaining your reasoning focusing on key factors like recent performance, matchup history, and value in the odds]

Consider factors like team recent performance, historical matchups, odds value, and market trends.'
                ),
                array(
                    'role' => 'user',
                    'content' => $prompt
                )
            ),
            'max_tokens' => 500,
            'temperature' => 0.7
        ));
        
        $response = wp_remote_post('https://api.openai.com/v1/chat/completions', array(
            'headers' => $headers,
            'body' => $body,
            'timeout' => 30
        ));
        
        if (is_wp_error($response)) {
            return 'Unable to generate AI analysis at this time.';
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['choices'][0]['message']['content'])) {
            return $body['choices'][0]['message']['content'];
        }
        
        return 'Unable to generate AI analysis at this time.';
    }
    
    private function get_best_odds_from_preferred_sportsbooks($bookmakers, $recommendation, $pick_type) {
        $preferred_sbooks = array('betmgm', 'fanduel', 'draftkings', 'caesars', 'espnbet');
        $sportsbook_names = array(
            'betmgm' => 'BetMGM',
            'fanduel' => 'FanDuel',
            'draftkings' => 'DraftKings',
            'caesars' => 'Caesars Sportsbook',
            'espnbet' => 'ESPN BET'
        );
        
        foreach ($preferred_sbooks as $preferred_book) {
            foreach ($bookmakers as $bookmaker) {
                if ($bookmaker['key'] === $preferred_book) {
                    foreach ($bookmaker['markets'] as $market) {
                        if ($market['key'] === 'h2h' && $pick_type === 'moneyline') {
                            foreach ($market['outcomes'] as $outcome) {
                                if ($outcome['name'] === $recommendation) {
                                    $decimal_odds = $outcome['price'];
                                    $american_odds = $this->convert_to_american_odds($decimal_odds);
                                    
                                    return array(
                                        'american_odds' => $american_odds,
                                        'sportsbook' => $bookmaker['key'],
                                        'sportsbook_display_name' => $sportsbook_names[$bookmaker['key']] ?? $bookmaker['title']
                                    );
                                }
                            }
                        }
                    }
                }
            }
        }
        
        return array(
            'american_odds' => null,
            'sportsbook' => null,
            'sportsbook_display_name' => null
        );
    }
    
    private function convert_to_american_odds($decimal_odds) {
        if ($decimal_odds >= 2.0) {
            return intval(($decimal_odds - 1) * 100);
        } else {
            return intval(-100 / ($decimal_odds - 1));
        }
    }
    
    private function is_game_today_or_tomorrow($game_time_str) {
        $game_time = strtotime($game_time_str);
        $today = strtotime('today');
        $tomorrow = strtotime('tomorrow');
        $day_after = strtotime('+2 days');
        
        return $game_time >= $today && $game_time < $day_after;
    }
    
    private function store_pick($pick) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'asbp_picks';
        
        $wpdb->insert(
            $table_name,
            array(
                'pick_id' => $pick['id'],
                'game_id' => $pick['game_id'],
                'sport_key' => $pick['sport_key'],
                'home_team' => $pick['home_team'],
                'away_team' => $pick['away_team'],
                'commence_time' => $pick['commence_time'],
                'pick_type' => $pick['pick_type'],
                'recommendation' => $pick['recommendation'],
                'confidence_level' => $pick['confidence_level'],
                'odds' => $pick['odds'],
                'sportsbook' => $pick['sportsbook'],
                'sportsbook_display_name' => $pick['sportsbook_display_name'],
                'analysis' => $pick['analysis'],
                'is_pick_of_day' => $pick['is_pick_of_day'] ? 1 : 0,
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%f', '%d', '%s', '%s', '%s', '%d', '%s')
        );
    }
    
    private function create_tables() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'asbp_picks';
        
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            pick_id varchar(50) NOT NULL,
            game_id varchar(100) NOT NULL,
            sport_key varchar(50) NOT NULL,
            home_team varchar(100) NOT NULL,
            away_team varchar(100) NOT NULL,
            commence_time datetime DEFAULT NULL,
            pick_type varchar(50) NOT NULL,
            recommendation varchar(100) NOT NULL,
            confidence_level decimal(3,2) NOT NULL,
            odds int DEFAULT NULL,
            sportsbook varchar(50) DEFAULT NULL,
            sportsbook_display_name varchar(100) DEFAULT NULL,
            analysis text NOT NULL,
            is_pick_of_day tinyint(1) DEFAULT 0,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY pick_id (pick_id),
            KEY sport_key (sport_key),
            KEY is_pick_of_day (is_pick_of_day),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    public function activate() {
        $this->create_tables();
        
        // Set default options
        add_option('asbp_odds_api_key', '');
        add_option('asbp_openai_api_key', '');
    }
    
    public function deactivate() {
        // Clean up transients
        delete_transient('asbp_pick_of_day_' . date('Y-m-d'));
    }
}

// Initialize the plugin
new AISportsBettingPicks();